﻿//-----------------------------------------------------------------------------
// MontyHallHelpScreen.cs
// (C) A. Schiffler, 2009-2011
//-----------------------------------------------------------------------------

namespace MontyHallGame
{
    using System;
    using System.Threading;
    using System.Collections.Generic;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;

    /// <summary>
    /// This screen implements the game help
    /// </summary>
    class MontyHallHelpScreen : GameScreen
    {

        /// <summary>
        /// The content manager.
        /// </summary>
        private ContentManager contentManager;

        // Text to display
        private List<string> messages;

        /// <summary>
        /// Number of text rows to fade in.
        /// </summary>
        private int numRows;

        /// <summary>
        /// Current row being faded.
        /// </summary>
        private int currentRow;

        /// <summary>
        /// The speed of the fade in alpha-change per update.
        /// </summary>
        private float fadeSpeed = 0.1f;

        /// <summary>
        /// Array of alpha values for each text row.
        /// </summary>
        private float[] textAlpha;

        #region Initialization

        /// <summary>
        /// Constructor of the screen.
        /// </summary>
        /// <param name="screenMessage">The help message to display.</param>
        public MontyHallHelpScreen(string[] screenMessage)
        {
            // New message list
            this.messages = new List<string>();

            // Store screen message
            this.messages.AddRange(screenMessage);

            // Add exit message
            string[] exitMessage = {"", "", "[Key] or (Button) to return to Main Menu"};
            this.messages.AddRange(exitMessage);

            TransitionOnTime = TimeSpan.FromSeconds(0.5);
            TransitionOffTime = TimeSpan.FromSeconds(0.5);
        }

        /// <summary>
        /// Load graphics content for the game.
        /// </summary>
        public override void LoadContent()
        {
            if (this.contentManager == null)
            {
                this.contentManager = new ContentManager(ScreenManager.Game.Services, "Content");
            }

            // Reset text fader 
            numRows = messages.Count;
            currentRow = 0;
            textAlpha = new float[numRows];
            for (int i = 0; i < numRows; i++)
            {
                textAlpha[i] = 0.0f;
            }

            // Reset game time
            ScreenManager.Game.ResetElapsedTime();
        }

        /// <summary>
        /// Unload graphics content used by the game.
        /// </summary>
        public override void UnloadContent()
        {
            this.contentManager.Unload();
        }

        #endregion

        #region Update and Draw

        /// <summary>
        /// Updates the state of the screen.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

            if (IsActive)
            {
                // Fade logic
                if (currentRow < numRows)
                {
                    textAlpha[currentRow] += fadeSpeed;
                    if (string.IsNullOrEmpty(messages[currentRow]) ||
                       (textAlpha[currentRow] >= 1.0f))
                    {
                        textAlpha[currentRow] = 1.0f;
                        currentRow++;
                    }
                }
            }
        }

        /// <summary>
        /// Lets the game respond to player input. Unlike the Update method,
        /// this will only be called when the gameplay screen is active.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            if (input == null)
            {
                throw new ArgumentNullException("input");
            }

            // Look up inputs for the active player profile.
            PlayerIndex playerIndex = ControllingPlayer.Value;

            KeyboardState keyboardState = input.CurrentKeyboardStates[(int)playerIndex];
            GamePadState gamePadState = input.CurrentGamePadStates[(int)playerIndex];

            // The game pauses either if the user presses the pause button, or if
            // they unplug the active gamepad. This requires us to keep track of
            // whether a gamepad was ever plugged in, because we don't want to pause
            // on PC if they are playing with a keyboard and have no gamepad at all!
            bool gamePadDisconnected = !gamePadState.IsConnected &&
                                       input.GamePadWasConnected[(int)playerIndex];

            // Back to main menu
            if (input.IsMenuSelect(ControllingPlayer, out playerIndex))
            {
                MontyHallLoadingScreen.Load(ScreenManager, false, null, new MontyHallBackgroundScreen(),
                                                               new MontyHallMainMenuScreen());
            }
        }

        /// <summary>
        /// Draws the gameplay screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            // Get drawing helpers and textures
            PrimitiveBatch primitiveBatch = ScreenManager.PrimitiveBatch;
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;
            SpriteFont font = ScreenManager.Fonts["game"];

            // Draw score summary
            Color color = Color.White;
            Vector2 origin = new Vector2(0, 0);
            Vector2 position = new Vector2(viewport.Width * 0.05f, viewport.Height * 0.05f);
            float scale = 0.5f;
            spriteBatch.Begin();
            for (int i=0; i<numRows; i++)
            {
                string message = messages[i];
                if (textAlpha[i] > 0.0f)
                {
                    if (string.IsNullOrEmpty(message))
                    {
                        // Skip this row
                        position.Y += ((float)font.LineSpacing * 0.5f * scale);
                    }
                    else
                    {
                        color = new Color(Color.White, MathHelper.SmoothStep(0.0f, 1.0f, textAlpha[i]));
                        spriteBatch.DrawString(
                            font,
                            message,
                            position,
                            color,
                            0,
                            origin,
                            scale,
                            SpriteEffects.None,
                            0);
                        position.Y += ((float)font.LineSpacing * 1.2f * scale);
                    }
                }
            }
            spriteBatch.End();

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }

            base.Draw(gameTime);
        }

        #endregion
    }
}
