//-----------------------------------------------------------------------------
// MontyHallMainMenuScreen.cs
// (C) A. Schiffler, 2009-2011
//-----------------------------------------------------------------------------

namespace MontyHallGame
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;
    using NewGamePhysics.StateManager;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.Networking;
    using NewGamePhysics.PhysicalElements;

    /// <summary>
    /// The main menu screen is the first thing displayed when the game starts up.
    /// </summary>
    class MontyHallMainMenuScreen : MenuScreen
    {
        /// <summary>
        /// Text scroller for informational message.
        /// </summary>
        private ScrollingText scrollingInfoText;

        /// <summary>
        /// Constructor fills in the menu contents.
        /// </summary>
        public MontyHallMainMenuScreen()
            : base("Monty Hall Game")
        {
            // Create our menu entries.
            MenuEntry hrngMenuEntry = new MenuEntry("Play with Physical Entropy");
            MenuEntry prngMenuEntry = new MenuEntry("Play with Simulated Entropy");
            MenuEntry optionsMenuEntry = new MenuEntry("Options and Settings");
            MenuEntry helpMenuEntry = new MenuEntry("Game Controls Help");
            MenuEntry infoMenuEntry = new MenuEntry("Game Information");
            MenuEntry exitMenuEntry = new MenuEntry("Exit Game");

            // Hook up menu event handlers.
            hrngMenuEntry.Selected += HrngMenuEntrySelected;
            prngMenuEntry.Selected += PrngMenuEntrySelected;
            optionsMenuEntry.Selected += OptionsMenuEntrySelected;
            infoMenuEntry.Selected += InfoMenuEntrySelected;
            helpMenuEntry.Selected += HelpMenuEntrySelected;
            exitMenuEntry.Selected += OnCancel;

            
            // Add entries to the menu;
            // randomize the users rng choices.
            Random random = new Random();
            if (random.Next(0, 2) == 0)
            {
                MenuEntries.Add(hrngMenuEntry);
                MenuEntries.Add(prngMenuEntry);
            }
            else
            {
                MenuEntries.Add(prngMenuEntry);
                MenuEntries.Add(hrngMenuEntry);
            }

            // Add other entries to menu
            MenuEntries.Add(optionsMenuEntry);
            MenuEntries.Add(infoMenuEntry);
            MenuEntries.Add(helpMenuEntry);
            MenuEntries.Add(exitMenuEntry);
        }

        public override void LoadContent()
        {
            // Create scrollers
            SpriteFont font = ScreenManager.Fonts["retro"];
            MontyHallGame game = (MontyHallGame)ScreenManager.Game;
            int width = game.GraphicsDevice.Viewport.Width;
            scrollingInfoText = new ScrollingText("", font, width, 420);
            scrollingInfoText.ScrollerSpeed = 125.0f;

            base.LoadContent();
        }


        /// <summary>
        /// Update the main menu screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        /// <param name="otherScreenHasFocus">
        /// Flag indicating of another screen has the focus.
        /// </param>
        /// <param name="coveredByOtherScreen">
        /// Flag indicating of the screen is covered by another screen.
        /// </param>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus, bool coveredByOtherScreen)
        {
            if (WaitForUncover)
            {
                if ((!coveredByOtherScreen) || (!otherScreenHasFocus))
                {
                    // Update the texts
                    UpdateAllTexts();

                    // Play jingle
                    ScreenManager.Sounds["intro"].Play();

                    // Update menu text
                    WaitForUncover = false;
                }
            }

            // Update scrollers
            scrollingInfoText.Update(gameTime);

            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);
        }

        /// <summary>
        /// Draw the main menu screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        public override void Draw(GameTime gameTime)
        {
            base.Draw(gameTime);

            // Draw scroller
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            scrollingInfoText.Draw(gameTime, spriteBatch);
        }

        #region Handle Input

        /// <summary>
        /// Event handler for when the Play Game menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void HrngMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Set the random number generator to a physical source
            MontyHallGame.state.Rng.EntropySource = EntropySourceType.PlayTrulyRandom;

            // Load game screen
            MontyHallLoadingScreen.Load(ScreenManager, true, e.PlayerIndex,
                               new MontyHallGameplayScreen());

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for when the Play Game menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void PrngMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Set the random number generator to a pseudo source
            MontyHallGame.state.Rng.EntropySource = EntropySourceType.Pseudo;

            // Load game screen
            MontyHallLoadingScreen.Load(ScreenManager, true, e.PlayerIndex,
                               new MontyHallGameplayScreen());

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for when the Options menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void OptionsMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            ScreenManager.AddScreen(new MontyHallOptionsMenuScreen(), e.PlayerIndex);

            // Reset back to play entry
            this.SelectedEntry = 0;

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for when the Info menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void InfoMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Text to display
            string[] messages = { 
            "Information on The Monty Game", 
            "",
            "The Monty Game is a game of chance which originated from the show",
            "\"Let's Make a Deal\" hosted by host Monty Hall (hence the name).",
            "",
            "A contestant is presented with 3 doors behind one of which a prize.",
            "In the 1st phase of the game, the contestant preselects one door.",            
            "This door is not opened, but the game show host will reveal a",
            "loosing pick in the 2nd phase by opening one of the remaining doors.",
            "In the 3rd and last phase of the game, the contestant can choose",
            "to open the original door or switch to  to claim the prize.",
            "",
            "So, what is the winning strategy? Stay with the preselected door,",
            "switch doors or does it actually matter ... play to find out.",
            };

            ScreenManager.AddScreen(new MontyHallHelpScreen(messages), e.PlayerIndex);

            // Reset back to play entry
            this.SelectedEntry = 0;

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// Event handler for when the Help menu entry is selected.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void HelpMenuEntrySelected(object sender, PlayerIndexEventArgs e)
        {
            // Text to display
            string[] messages = { 
            "Controls for The Monty Game", 
            "",
            " [Up]/[Down] cursor/pad",
            "            select menu item",
            " [Left]/[Right] cursor/pad",
            "            select available door to the left/right",
            " [Space] or (A)",
            "            continue in menu, game or help screens",
            "            Phase 1: pre-select a door",
            "            Phase 2: let Monty open a loosing door",
            "            Phase 3: open door to claim a prize",
            " [Escape] or (Back)",
            "            pause game or quit to main menu",
            };

            ScreenManager.AddScreen(new MontyHallHelpScreen(messages), e.PlayerIndex);

            // Reset back to play entry
            this.SelectedEntry = 0;

            // Set flag to update text when we return
            WaitForUncover = true;
        }

        /// <summary>
        /// When the user cancels the main menu, ask if they want to exit the sample.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        protected override void OnCancel(PlayerIndex playerIndex)
        {
            const string message = "Are you sure you want to exit?";

            MontyHallMessageBoxScreen confirmExitMessageBox = new MontyHallMessageBoxScreen(message);

            confirmExitMessageBox.Accepted += ConfirmExitMessageBoxAccepted;

            ScreenManager.AddScreen(confirmExitMessageBox, playerIndex);
        }

        /// <summary>
        /// Event handler for when the user selects ok on the "are you sure
        /// you want to exit" message box.
        /// </summary>
        /// <param name="sender">The event sender</param>
        /// <param name="e">The event args.</param>
        void ConfirmExitMessageBoxAccepted(object sender, PlayerIndexEventArgs e)
        {
            ScreenManager.Game.Exit();
        }

        #endregion

        /// <summary>
        /// Update menu and scroller text.
        /// </summary>
        private void UpdateAllTexts()
        {
            // Scroller text
            if (null != scrollingInfoText)
            {
                scrollingInfoText.Text =
                    "The Monty Hall Game (C) A. Schiffler 2009-2011, aschiffler@ferzkopp.net - source code is available upon request." +
                    "  *  " +
                    "This game tracks some anonymous usage statistics as part of the authors PhD research into 'New Game Physics'. If you don't agree to this, please quit now." +
                    "  *  " +
                    "Physical entropy is provided by the playtrulyrandom.com webservice. Please navigate to the website for more technical and scientific information." +
                    "  *  " +
                    "Simulated entropy provided by the internal C# System.Random class. Please check the Microsoft MSDN website for more information.";
            }
        }
    }
}
