//-----------------------------------------------------------------------------
// MontyHallOptionsMenuScreen.cs
// (C) A. Schiffler, 2009-2011
//-----------------------------------------------------------------------------

namespace MontyHallGame
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    
    /// <summary>
    /// The options for the Monty Hall Game.
    /// </summary>
    class MontyHallOptionsMenuScreen : MenuScreen
    {
        /// <summary>
        /// Text scroller for informational message.
        /// </summary>
        private ScrollingText scrollingInfoText;

        #region constructor

        /// <summary>
        /// Constructor for the options menu screen of the gravity chooser.
        /// </summary>
        public MontyHallOptionsMenuScreen()
            : base("Monty Hall Game Options")
        {
            MenuEntry backMenuEntry = new MenuEntry("Back to Main Menu");
            backMenuEntry.Selected += OnCancel;
            MenuEntries.Add(backMenuEntry);
        }

        #endregion

        public override void LoadContent()
        {
            base.LoadContent();

            // Create scroller
            SpriteFont font = ScreenManager.Fonts["menu"];
            int width = ScreenManager.GraphicsDevice.Viewport.Width;
            scrollingInfoText = new ScrollingText("-", font, width, 400); // text will be set later
            scrollingInfoText.ScrollerSpeed = 200.0f;
        }

        /// <summary>
        /// Update the main menu screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        /// <param name="otherScreenHasFocus">
        /// Flag indicating of another screen has the focus.
        /// </param>
        /// <param name="coveredByOtherScreen">
        /// Flag indicating of the screen is covered by another screen.
        /// </param>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus, bool coveredByOtherScreen)
        {
            if (WaitForUncover)
            {
                if ((!coveredByOtherScreen) || (!otherScreenHasFocus))
                {
                    // Update the texts
                    UpdateAllTexts();

                    // Update menu text
                    WaitForUncover = false;
                }
            }

            // Update scrollers
            scrollingInfoText.Update(gameTime);

            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);
        }

        /// <summary>
        /// Draw the options screen.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        public override void Draw(GameTime gameTime)
        {
            base.Draw(gameTime);

            // Draw scroller
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            scrollingInfoText.Draw(gameTime, spriteBatch);
        }

        #region private_methods

        /// <summary>
        /// Fills in the options screen texts from the game state.
        /// </summary>
        private void UpdateAllTexts()
        {
            // Scroller text
            if (null != scrollingInfoText)
            {
                scrollingInfoText.Text = "No options yet.";
            }
        }

        #endregion

        #region event_handlers

        // None yet.

        #endregion
    }
}
