﻿
using System;
using System.IO;
using System.IO.Ports;
using System.Diagnostics;
using System.Reflection;
using System.Runtime.InteropServices;
using Microsoft.Win32.SafeHandles;

using NewGamePhysics.Utilities;

namespace NewGamePhysics.Devices
{
    /// <summary>
    /// Callback that handles a geiger counter event.
    /// </summary>
    /// <param name="relativeTime"></param>
    public delegate void GeigerCounterEvent(double relativeTime);

    /// <summary>
    /// Class to interface the AWARE electronics RM-60, RM-70 and RM-80
    /// serial geiger counters.
    /// </summary>
    public class AwareGeigerCounter : IDisposable
    {
        /// <summary>
        /// Indicates that ring bit is on (winbase.h modem constant). 
        /// </summary>
        private const UInt32 MS_RING_ON = 0x0040;

        /// <summary>
        /// Callback handler for measurements.
        /// </summary>
        private GeigerCounterEvent eventHandler;

        /// <summary>
        /// The serial port the device is connected to.
        /// </summary>
        private SerialPort serialPort;

        /// <summary>
        /// Pointer to the Win32 HANDLE for the device.
        /// </summary>
        private IntPtr serialPortHandle;

        /// <summary>
        /// Flag indicating that the last ring bit is set/on.
        /// Used on downward-edge detection from counter.
        /// </summary>
        private bool lastRingOn = false;

        /// <summary>
        /// Initialize an AWARE geiger counter.
        /// </summary>
        /// <param name="portReference">The name of the serial port to connect to (i.e. COM1)
        /// or a number to use as index to available ports (i.e. 2 = second available port).</param>
        public AwareGeigerCounter(string portReference)
        {
            if (string.IsNullOrEmpty(portReference))
            {
                throw new ArgumentNullException(
                    "portReference",
                    "The port reference cannot be null or empty");
            }

            string[] portNames = System.IO.Ports.SerialPort.GetPortNames();
            string portName;

            if ((portNames == null) || (portNames.Length == 0))
            {
                throw new ApplicationException(
                    "No serial ports are available on this system");
            }

            if (!Array.Exists(portNames, s => s.ToUpper().Contains(portReference.ToUpper())))
            {
                try
                {
                    int portIndex = Convert.ToInt32(portReference);

                    if ((portIndex < 1) || (portIndex > portNames.Length))
                    {
                        SerialPortInfo(portNames);
                        throw new ArgumentOutOfRangeException(
                            "portIndex",
                            "The port index must be in the range 1 to " + portNames.Length);
                    }

                    portName = portNames[portIndex - 1];
                }
                catch (Exception)
                {
                    SerialPortInfo(portNames);
                    throw;
                }
            }
            else
            {
                portName = portReference;
            }

            this.InitializePort(portName);
        }

        /// <summary>
        /// Dispose of the AwareGeigerCounter object by removing event handler 
        /// and closing port.
        /// </summary>
        public void Dispose()
        {
            if (this.serialPort != null)
            {
                this.serialPortHandle = (IntPtr)0;
                this.serialPort.PinChanged -= new SerialPinChangedEventHandler(PinChanged);
                this.serialPort.Close();
                this.serialPort.Dispose();
                this.serialPort = null;
            }
        }

        /// <summary>
        /// Destructor for AwareGeigerCounter object.
        /// </summary>
        ~AwareGeigerCounter()
        {
            this.Dispose();
        }

        /// <summary>
        /// Sets the event handler which is called
        /// for each detection event.
        /// </summary>
        public GeigerCounterEvent EventHandler
        {
            set { eventHandler = value; }
        }
        /// <summary>
        /// Prints info of valid serial ports.
        /// </summary>
        private void SerialPortInfo(string[] portNames)
        {
            System.Console.WriteLine();
            System.Console.WriteLine("Available Serial Ports:");
            for (int i = 0; i < portNames.Length; i++)
            {
                System.Console.WriteLine(" port #" + (i + 1) + " on: " + portNames[i]);
            }
            System.Console.WriteLine();
        }

        /// <summary>
        /// Initializes the serial port and connects the event callback.
        /// </summary>
        /// <param name="portName">The port to initialize.</param>
        private void InitializePort(string portName)
        {
            // New serial port
            this.serialPort = new SerialPort(portName, 2400);

            // Power the device
            this.serialPort.Open();
            this.serialPort.DtrEnable = true;
            this.serialPort.RtsEnable = false;

            // Get a Win32 handle to serial port
            FieldInfo internalStreamFieldInfo = typeof(SerialPort).GetField(
                "internalSerialStream",
                BindingFlags.NonPublic | BindingFlags.Instance);
            object internalStream = internalStreamFieldInfo.GetValue(this.serialPort);
            Type internalStreamType = internalStream.GetType();
            FieldInfo handleFieldInfo = internalStreamType.GetField(
                "_handle",
                BindingFlags.NonPublic | BindingFlags.Instance);
            object safeFileHandle = handleFieldInfo.GetValue(internalStream);
            this.serialPortHandle = ((SafeFileHandle)safeFileHandle).DangerousGetHandle();

            // Connect event to detect counts on RI pin
            this.serialPort.PinChanged += new SerialPinChangedEventHandler(PinChanged);
        }

        /// <summary>
        /// Retrieves the modem control-register values.
        /// </summary>
        /// <param name="hFile">Handle to the communications device.</param>
        /// <param name="lpModemStat">Returned control-register status.</param>
        /// <returns>Non-zero if function succeeds.</returns>
        [DllImport("kernel32.dll")]
        private static extern Boolean GetCommModemStatus(IntPtr hFile, out UInt32 lpModemStat);

        /// <summary>
        /// Event handler for the PinChanged event. Calls event handler if downward edge
        /// of the serial ports RING bit was detected.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event parameters.</param>
        private void PinChanged(object sender, SerialPinChangedEventArgs e)
        {
            if (e.EventType == SerialPinChange.Ring)
            {
                // Gate event by checking for down edge of ring bit change
                uint status;
                if (GetCommModemStatus(this.serialPortHandle, out status))
                {
                    bool ringOn = (status & MS_RING_ON) != 0;
                    if ((this.lastRingOn) && (!ringOn))
                    {
                        if (null != this.eventHandler)
                        {
                            this.eventHandler(HighResolutionTimer.Seconds());
                        }
                    }

                    this.lastRingOn = ringOn;
                }
                else
                {
                    string message = "GetCommModemStatus() error: " +
                        Marshal.GetLastWin32Error().ToString();
                    throw new ApplicationException(message);
                }
            }
        }
    }
}
