﻿//-----------------------------------------------------------------------------
// BloomOverlay.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.Mathematics;

    /// <summary>
    /// 
    /// </summary>
    public class BloomOverlay : GraphicalElementBase
    {
        /// <summary>
        /// The effect to extract bright parts.
        /// </summary>
        private Effect extractEffect;

        /// <summary>
        /// The effect to blur (gaussian filter).
        /// </summary>
        private Effect filterEffect;

        /// <summary>
        /// The effect to combine the new and original image.
        /// </summary>
        private Effect combineEffect;

        /// <summary>
        /// The backbuffer.
        /// </summary>
        private ResolveTexture2D resolveTarget;

        /// <summary>
        /// Work texture 1
        /// </summary>
        private RenderTarget2D renderTarget1;

        /// <summary>
        /// Work texture 2
        /// </summary>
        private RenderTarget2D renderTarget2;

        /// <summary>
        /// Precalculated Gaussian tap weight values.
        /// </summary>
        float[] tapWeights;

        /// <summary>
        /// Precalculated Horizontal tap offsets.
        /// </summary>
        Vector2[] horizontalTapOffsets;

        /// <summary>
        /// Precalculated Vertical tap offsets.
        /// </summary>
        Vector2[] verticalTapOffsets;

        /// <summary>
        /// Filter parameter to set tap weights.
        /// </summary>
        EffectParameter weightsParameter;

        /// <summary>
        /// Filter parameter to set tap offsets.
        /// </summary>
        EffectParameter offsetsParameter;

        /// <summary>
        /// Creates a bloom overay.
        /// </summary>
        /// <param name="screenManager">The screen manager to use.</param>
        public BloomOverlay(
            ScreenManager screenManager) : base(screenManager)
        {
        }

        /// <summary>
        /// Load effects, create textures and prepare filters.
        /// </summary>
        public void LoadContent()
        {
            this.extractEffect = 
                this.ScreenManager.Game.Content.Load<Effect>("Effects/ThresholdAndRescale");
            this.filterEffect = 
                this.ScreenManager.Game.Content.Load<Effect>("Effects/TapFilter15");
            this.combineEffect =
                this.ScreenManager.Game.Content.Load<Effect>("Effects/Combine");

            // Look up the resolution and format of our main backbuffer.
            PresentationParameters presentationParams = 
                this.ScreenManager.GraphicsDevice.PresentationParameters;
            int width = presentationParams.BackBufferWidth;
            int height = presentationParams.BackBufferHeight;
            SurfaceFormat format = presentationParams.BackBufferFormat;

            // Create a texture for reading back the backbuffer contents.
            this.resolveTarget = new ResolveTexture2D(
                this.ScreenManager.GraphicsDevice,
                width,
                height,
                1,
                format);

            // Create half-size bloom render targets
            width /= 2;
            height /= 2;
            renderTarget1 = new RenderTarget2D(
                this.ScreenManager.GraphicsDevice, 
                width, 
                height, 
                1,
                format,
                this.ScreenManager.GraphicsDevice.DepthStencilBuffer.MultiSampleType,
                this.ScreenManager.GraphicsDevice.PresentationParameters.MultiSampleQuality);
            renderTarget2 = new RenderTarget2D(
                this.ScreenManager.GraphicsDevice, 
                width, 
                height, 
                1,
                format,
                this.ScreenManager.GraphicsDevice.DepthStencilBuffer.MultiSampleType,
                this.ScreenManager.GraphicsDevice.PresentationParameters.MultiSampleQuality);

            // Get filter parameters
            this.weightsParameter = this.filterEffect.Parameters["TapWeights"];
            this.offsetsParameter = this.filterEffect.Parameters["TapOffsets"];

            // Calculate steps in filter
            int tapCount = weightsParameter.Elements.Count;
            int tapSteps = tapCount / 2;

            // Calculate tap filter weights and offsets
            this.tapWeights = new float[tapCount];
            this.horizontalTapOffsets = new Vector2[tapCount];
            this.verticalTapOffsets = new Vector2[tapCount];
            float dx = 1.0f / width;
            float dy = 1.0f / height;
            double size = 2.0;
            float weightSum = (float)Gaussian.DistributionValue(size, 0.0);
            this.tapWeights[0] = weightSum;
            this.verticalTapOffsets[0] = new Vector2();
            this.horizontalTapOffsets[0] = new Vector2();
            for (int i = 0; i < tapSteps; i++)
            {
                // Store weights for the positive and negative taps.
                float weight = (float)Gaussian.DistributionValue(size, (double)(i + 1));
                int left = i * 2 + 1;
                int right = i * 2 + 2;
                this.tapWeights[right] = weight;
                this.tapWeights[left] = weight;
                weightSum += (2.0f * weight);

                // Store Offsets
                float sampleOffset = right + 0.5f;
                this.horizontalTapOffsets[right] = new Vector2(dx, 0.0f) * sampleOffset;
                this.horizontalTapOffsets[left] = new Vector2(-dx, 0.0f) * sampleOffset;
                this.verticalTapOffsets[right] = new Vector2(0.0f, dy) * sampleOffset;
                this.verticalTapOffsets[left] = new Vector2(0.0f, -dy) * sampleOffset;
            }

            // Normalize weights
            for (int i = 1; i < tapCount; i++)
            {
                this.tapWeights[i] /= weightSum;
            }
        }

        /// <summary>
        /// Unload textures.
        /// </summary>
        public void UnloadContent()
        {
            resolveTarget.Dispose();
            renderTarget1.Dispose();
            renderTarget2.Dispose();
        }

        /// <summary>
        /// This is where it all happens. Grabs a scene that has already been rendered,
        /// and uses postprocess magic to add a glowing bloom effect over the top of it.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        public void Draw(GameTime gameTime)
        {
            // Resolve the scene into a texture, so we can
            // use it as input data for the bloom processing.
            this.ScreenManager.GraphicsDevice.ResolveBackBuffer(resolveTarget);

            // Pass 1: draw the scene into rendertarget 1, using a
            // shader that extracts only the brightest parts of the image.
            this.extractEffect.Parameters["threshold"].SetValue(0.0f);
            DrawQuadWithEffect(resolveTarget, renderTarget1, this.extractEffect);

            // Pass 2: draw from rendertarget 1 into rendertarget 2,
            // using a shader to apply a horizontal gaussian blur filter.
            // Look up the sample weight and offset effect parameters.
            this.weightsParameter.SetValue(tapWeights);
            this.offsetsParameter.SetValue(horizontalTapOffsets);
            DrawQuadWithEffect(renderTarget1.GetTexture(), renderTarget2, this.filterEffect);
        
            // Pass 3: draw from rendertarget 2 back into rendertarget 1,
            // using a shader to apply a vertical gaussian blur filter.
            this.weightsParameter.SetValue(tapWeights);
            this.offsetsParameter.SetValue(verticalTapOffsets);
            DrawQuadWithEffect(renderTarget2.GetTexture(), renderTarget1, this.filterEffect);

            // Pass 4: draw both rendertarget 1 and the original scene
            // image back into the main backbuffer, using a shader that
            // combines them to produce the final bloomed result
            this.ScreenManager.GraphicsDevice.Textures[1] = resolveTarget;
            this.combineEffect.Parameters["CombineIntensity"].SetValue(2.0f);
            Viewport viewport = this.ScreenManager.GraphicsDevice.Viewport;
            this.ScreenManager.SpriteBatch.Begin(
                SpriteBlendMode.None,
                SpriteSortMode.Immediate,
                SaveStateMode.None);
            this.combineEffect.Begin();
            this.combineEffect.CurrentTechnique.Passes[0].Begin();
            this.ScreenManager.SpriteBatch.Draw(
                renderTarget1.GetTexture(),
                new Rectangle(0, 0, viewport.Width, viewport.Height),
                Color.White);
            this.ScreenManager.SpriteBatch.End();
            this.combineEffect.CurrentTechnique.Passes[0].End();
            this.combineEffect.End();
        }

        /// <summary>
        /// Draw a texture full-screen into a rendertarget using a custom shader.
        /// </summary>
        /// <param name="texture">The texture to draw.</param>
        /// <param name="renderTarget">The render target.</param>
        /// <param name="effect">The effect to apply.</param>
        private void DrawQuadWithEffect(
            Texture2D texture, 
            RenderTarget2D renderTarget,
            Effect effect)
        {
            this.ScreenManager.GraphicsDevice.SetRenderTarget(0, renderTarget);
            this.ScreenManager.SpriteBatch.Begin(
                SpriteBlendMode.None,
                SpriteSortMode.Immediate,
                SaveStateMode.None);
            effect.Begin();
            effect.CurrentTechnique.Passes[0].Begin();
            this.ScreenManager.SpriteBatch.Draw(
                texture,
                new Rectangle(0, 0, renderTarget.Width, renderTarget.Height),
                Color.White);
            this.ScreenManager.SpriteBatch.End();
            effect.CurrentTechnique.Passes[0].End();
            effect.End();
            this.ScreenManager.GraphicsDevice.SetRenderTarget(0, null);
        }
    }
}
