﻿

namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Animated and drawable double pendulum.
    /// </summary>
    public class DoubleRegularPendulum : GraphicalElementBase
    {
        /// <summary>
        /// The pendulum simulation.
        /// </summary>
        DoubleRegularPendulumSimulation pendulum;

        /// <summary>
        /// The origin for mapping simulation space to the screen.
        /// </summary>
        Vector2 screenOrigin;

        /// <summary>
        /// The scale for mapping simulation space to the screen.
        /// </summary>
        double screenScale;

        /// <summary>
        /// Texture for the hinge rivet.
        /// </summary>
        private static Texture2D rivetTexture;

        /// <summary>
        /// Constructor for animated double pendulum.
        /// </summary>
        /// <param name="screenManager">The screen manager to use for drawing.</param>
        /// <param name="simulationOrigin">The origin for pendulum in simulation space.</param>
        /// <param name="l1">Length of the first rod in simulation space.</param>
        /// <param name="m1">Mass of the first pendulum component.</param>
        /// <param name="l2">Length of the second rod in simulation space.</param>
        /// <param name="m2">Mass of the second pendulum component</param>
        /// <param name="g">The gravitational acceleration to apply.</param>
        /// <param name="f">The frictional model to use for the hinges.</param>
        /// <param name="screenOrigin">The origin for mapping simulation space to the screen.</param>
        /// <param name="screenScale">The scale for mapping simulation space to the screen.</param>
        public DoubleRegularPendulum(
            ScreenManager screenManager,
            Vector2 simulationOrigin,
            double l1,
            double m1,
            double l2,
            double m2,
            double g,
            RotationalFrictionType f, 
            Vector2 screenOrigin,
            double screenScale)
            : base(screenManager)
        {
            // Create pendulum simulation
            this.pendulum = new DoubleRegularPendulumSimulation(simulationOrigin, l1, m1, l2, m2, g, f);
            this.screenOrigin = screenOrigin;
            this.screenScale = screenScale;

            // Prepare texture
            if (this.ScreenManager.Textures.ContainsKey("rivet"))
            {
                rivetTexture = this.ScreenManager.Textures["rivet"];
            }
        }

        /// <summary>
        /// Gets the pendulum simulation.
        /// </summary>
        public DoubleRegularPendulumSimulation Pendulum
        {
            get { return this.pendulum; }
        }

        /// <summary>
        /// Draw the double pendulum.
        /// </summary>
        /// <param name="gameTime">Current game time.</param>
        /// <param name="pendulum">The pendulum object.</param>
        public void Draw(GameTime gameTime)
        {
            // Draw Pendulum
            Vector2[] pendulumPoints = this.pendulum.GetPosition(this.screenOrigin, this.screenScale);
            this.PrimitiveBatch.Begin(PrimitiveType.LineList);
            this.PrimitiveBatch.AddVertex(pendulumPoints[0], Color.White);
            this.PrimitiveBatch.AddVertex(pendulumPoints[1], Color.White);
            this.PrimitiveBatch.AddVertex(pendulumPoints[1], Color.White);
            this.PrimitiveBatch.AddVertex(pendulumPoints[2], Color.White);
            this.PrimitiveBatch.End();

            // Check if we have the texture
            if (null != rivetTexture)
            {
                // Rivets
                Rectangle dest = new Rectangle(0, 0, 8, 8);
                this.SpriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                dest.X = Convert.ToInt32(Math.Round(pendulumPoints[0].X)) - 4;
                dest.Y = Convert.ToInt32(Math.Round(pendulumPoints[0].Y)) - 4;
                this.SpriteBatch.Draw(rivetTexture, dest, Color.White);
                dest.X = Convert.ToInt32(Math.Round(pendulumPoints[1].X)) - 8;
                dest.Y = Convert.ToInt32(Math.Round(pendulumPoints[1].Y)) - 8;
                dest.Width = 16;
                dest.Height = 16;
                this.SpriteBatch.Draw(rivetTexture, dest, Color.White);
                dest.X = Convert.ToInt32(Math.Round(pendulumPoints[2].X)) - 8;
                dest.Y = Convert.ToInt32(Math.Round(pendulumPoints[2].Y)) - 8;
                this.SpriteBatch.Draw(rivetTexture, dest, Color.White);
                this.SpriteBatch.End();
            }
        }
    }
}

