﻿

namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.PhysicalElements;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;

    /// <summary>
    /// Animated and drawable double square pendulum.
    /// </summary>
    public class DoubleSquarePendulum : GraphicalElementBase
    {
        /// <summary>
        /// The pendulum simulation.
        /// </summary>
        DoubleSquarePendulumSimulation pendulum;

        /// <summary>
        /// The origin for mapping simulation space to the screen.
        /// </summary>
        Vector2 screenOrigin;

        /// <summary>
        /// The scale for mapping simulation space to the screen.
        /// </summary>
        double screenScale;

        /// <summary>
        /// Texture for the pendulum block.
        /// </summary>
        private static Texture2D blockTexture;

        /// <summary>
        /// Texture for the hinge rivet.
        /// </summary>
        private static Texture2D rivetTexture;

        /// <summary>
        /// Create an animated double square pendulum.
        /// </summary>
        /// <param name="simulationOrigin">The origin for pendulum in simulation space.</param>
        /// <param name="l">The length of the side of each square in simulation space.</param>
        /// <param name="m">The mass of each square.</param>
        /// <param name="g">The gravitational acceleration to apply.</param>
        /// <param name="f">The friction model to use for the hinges.</param>
        /// <param name="screenOrigin">The origin for mapping simulation space to the screen.</param>
        /// <param name="screenScale">The scale for mapping simulation space to the screen.</param>
        public DoubleSquarePendulum(
            ScreenManager screenManager,
            Vector2 simulationOrigin,
            double l,
            double m,
            double g,
            RotationalFrictionType f,
            Vector2 screenOrigin,
            double screenScale)
            : base(screenManager)
        {
            // Create pendulum simulation
            this.pendulum = new DoubleSquarePendulumSimulation(simulationOrigin, l, m, g, f);
            this.screenOrigin = screenOrigin;
            this.screenScale = screenScale;

            // Prepare textures
            if (this.ScreenManager.Textures.ContainsKey("rivet"))
            {
                rivetTexture = this.ScreenManager.Textures["rivet"];
            }

            if (this.ScreenManager.Textures.ContainsKey("metal"))
            {
                blockTexture = this.ScreenManager.Textures["metal"];
            }

        }

        /// <summary>
        /// Gets the pendulum simulation.
        /// </summary>
        public DoubleSquarePendulumSimulation Pendulum
        {
            get { return this.pendulum; }
        }

        /// <summary>
        /// Draws the double square pendulum.
        /// </summary>
        /// <param name="gameTime">Current game time.</param>
        public void Draw(GameTime gameTime)
        {
            // Get physical state of pendulum
            double[] pendulumSizes = this.pendulum.GetSize(this.screenScale);
            double[] pendulumAngles = this.pendulum.GetAngle();
            Vector2[] pendulumPoints = this.pendulum.GetPosition(this.screenOrigin, this.screenScale);

            // Temp var
            Rectangle dest = new Rectangle();

            // Draw top box
            if (null != blockTexture)
            {
                this.SpriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                dest = new Rectangle();
                dest.X = Convert.ToInt32(Math.Round(pendulumPoints[0].X));
                dest.Y = Convert.ToInt32(Math.Round(pendulumPoints[0].Y));
                dest.Width = (int)pendulumSizes[0];
                dest.Height = dest.Width;
                this.SpriteBatch.Draw(
                    blockTexture,
                    dest,
                    null,
                    Color.White,
                    (float)(Math.PI / 4 - pendulumAngles[0]),
                    new Vector2(),
                    SpriteEffects.None,
                    (float)0.0);
                this.SpriteBatch.End();
            }

            // Draw top frame
            this.PrimitiveBatch.Begin(PrimitiveType.LineList);
            this.PrimitiveBatch.AddVertex(pendulumPoints[0], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[1], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[1], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[2], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[2], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[3], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[3], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[0], Color.Black);
            this.PrimitiveBatch.End();

            // Draw top rivet
            if (null != rivetTexture)
            {
                this.SpriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                dest.X -= 4;
                dest.Y -= 4;
                dest.Width = 8;
                dest.Height = 8;
                this.SpriteBatch.Draw(
                    rivetTexture,
                    dest,
                    Color.White);
                this.SpriteBatch.End();
            }

            // Draw bottom box
            if (null != blockTexture)
            {
                this.SpriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                dest.X = Convert.ToInt32(Math.Round(pendulumPoints[4].X));
                dest.Y = Convert.ToInt32(Math.Round(pendulumPoints[4].Y));
                dest.Width = (int)pendulumSizes[1];
                dest.Height = dest.Width;
                this.SpriteBatch.Draw(
                    blockTexture,
                    dest,
                    null,
                    Color.White,
                    (float)(Math.PI / 4 - pendulumAngles[1]),
                    new Vector2(),
                    SpriteEffects.None,
                    (float)0.0);
                this.SpriteBatch.End();
            }

            // Draw bottom frame
            this.PrimitiveBatch.Begin(PrimitiveType.LineList);
            this.PrimitiveBatch.AddVertex(pendulumPoints[4], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[5], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[5], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[6], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[6], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[7], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[7], Color.Black);
            this.PrimitiveBatch.AddVertex(pendulumPoints[4], Color.Black);
            this.PrimitiveBatch.End();

            // Draw bottom rivet
            if (null != rivetTexture)
            {
                this.SpriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                dest.X -= 4;
                dest.Y -= 4;
                dest.Width = 8;
                dest.Height = 8;
                this.SpriteBatch.Draw(
                    rivetTexture,
                    dest,
                    Color.White);
                this.SpriteBatch.End();
            }
        }
    }
}


