﻿using System;
using System.Collections.Generic;
using System.Text;

using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

using NewGamePhysics.StateManager;
using NewGamePhysics.Utilities;

namespace NewGamePhysics.GraphicalElements
{
    /// <summary>
    /// A energy indicator color bar for a floating point value in 
    /// the range 0 - 100 (i.e. percentage). Includes optional 
    /// one character label.
    /// </summary>
    public class EnergyIndicator : GraphicalElementBase
    {
        /// <summary>
        /// The width of the indicator in pixels.
        /// </summary>
        public const float Width = 14.0f;

        /// <summary>
        /// The height of the indicator in pixels.
        /// </summary>
        public const float Height = 102.0f;

        /// <summary>
        /// The current value of the indicator.
        /// </summary>
        private double value;

        /// <summary>
        /// The position where the indicator is drawn on the screen.
        /// </summary>
        private Vector2 position;

        /// <summary>
        /// The text of the indicator.
        /// </summary>
        private string label;

        /// <summary>
        /// The color of the indicator box and the font.
        /// </summary>
        private Color drawColor;

        /// <summary>
        /// Black texture for hiding parts of the indicator.
        /// </summary>
        private Texture2D blackTexture;

        /// <summary>
        /// Colored bar texture for drawing the bar graph of the indicator.
        /// </summary>
        private Texture2D barTexture;

        /// <summary>
        /// The font for the label.
        /// </summary>
        private SpriteFont textFont; 

        /// <summary>
        /// Create new indicator object for use with autosizing.
        /// </summary>
        /// <param name="screenManager">The screen manager to use for drawing the indicator.</param>
        /// <param name="label">The label of the indicator.</param>
        public EnergyIndicator(
            ScreenManager screenManager,
            string label) : base(screenManager)
        {
            // Store settings
            this.label = label;

            // Default position and size
            this.position = new Vector2();

            // Black hide texture
            this.blackTexture = TextureHelpers.Create(screenManager.GraphicsDevice, Color.Black);

            // Colored bar texture
            this.barTexture = screenManager.Game.Content.Load<Texture2D>(@"Sprites\energybar");

            // Color for box and font
            this.drawColor = new Color(255, 255, 255);

            // The font to use
            this.textFont = ScreenManager.Fonts["small"];

            // Now set a default value, set bar color.
            this.SetValue(0.0);
        }

        /// <summary>
        /// Sets a new value, but limit range to 0-100.
        /// </summary>
        /// <param name="value">The value to set.</param>
        public void SetValue(double value)
        {
            // Set value but limit range
            if (value < 0.0)
            {
                this.value = 0.0;
            }
            else if (value > 100.0)
            {
                this.value = 100.0;
            }
            else
            {
                this.value = value;
            }
        }

        /// <summary>
        /// Reposition indicator object.
        /// </summary>
        /// <param name="position"></param>
        public void SetPosition(Vector2 position)
        {
            this.position.X = position.X;
            this.position.Y = position.Y;
        }

        /// <summary>
        /// Draws the indicator.
        /// </summary>
        /// <param name="gameTime">The game time for drawing.</param>
        public void Draw(GameTime gameTime)
        {
            Rectangle destination;

            // Draw bar
            destination = new Rectangle(
                (int)this.position.X, 
                (int)this.position.Y,
                (int)Width,
                (int)Height);
            SpriteBatch.Begin();
            SpriteBatch.Draw(this.barTexture, destination, Color.White);
            SpriteBatch.End();

            // Hide invisible part of bar
            int hiddenHeight = (int)((100.0 - this.value) * 0.01 * (double)Height); 
            destination = new Rectangle(
                (int)this.position.X, 
                (int)this.position.Y, 
                (int)Width, 
                (int)hiddenHeight);
            SpriteBatch.Begin();
            SpriteBatch.Draw(this.blackTexture, destination, Color.DarkSlateGray);
            SpriteBatch.End();

            // Lines
            PrimitiveBatch.Begin(PrimitiveType.LineList);

            // Box
            PrimitiveBatch.AddVertex(this.position, this.drawColor);
            PrimitiveBatch.AddVertex(this.position + new Vector2(Width, 0.0f), this.drawColor);

            PrimitiveBatch.AddVertex(this.position + new Vector2(Width, 0.0f), this.drawColor);
            PrimitiveBatch.AddVertex(this.position + new Vector2(Width, Height), this.drawColor);

            PrimitiveBatch.AddVertex(this.position + new Vector2(Width, Height), this.drawColor);
            PrimitiveBatch.AddVertex(this.position + new Vector2(0.0f, Height), this.drawColor);

            PrimitiveBatch.AddVertex(this.position + new Vector2(0.0f, Height), this.drawColor);
            PrimitiveBatch.AddVertex(this.position, this.drawColor);

            PrimitiveBatch.End();

            // Label (sits on top of visible area)
            if (!String.IsNullOrEmpty(this.label))
            {
                float ySpacing = 2.0f;
                Vector2 textSize = this.textFont.MeasureString(this.label);
                int visibleHeight = (int)(this.value * 0.01 * (double)Height);
                Vector2 textPosition = new Vector2();
                textPosition.X = this.position.X + (Width - textSize.X) / 2;
                float delta = ySpacing + visibleHeight + textSize.Y;
                if (delta > (Height - ySpacing))
                {
                    textPosition.Y = this.position.Y + ySpacing;
                }
                else
                {
                    textPosition.Y = this.position.Y + Height - delta;
                }

                SpriteBatch.Begin();
                SpriteBatch.DrawString(
                    this.textFont,
                    this.label,
                    textPosition,
                    this.drawColor, 0, new Vector2(), 1.0f,
                    SpriteEffects.None, 0);
                SpriteBatch.End();
            }
        }
    }
}
