﻿using System;
using System.Threading;
using System.Collections.Generic;

using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;

using NewGamePhysics.StateManager;
using NewGamePhysics.Utilities;

namespace NewGamePhysics.GraphicalElements
{
    /// <summary>
    /// Enumeration for the text message alignment.
    /// </summary>
    public enum InfoMessageAlignment
    {
        /// <summary>
        /// Left-align text lines at X of origin.
        /// i.e. origin is the upper left corner of the text block.
        /// </summary>
        Left,

        /// <summary>
        /// Center-align text lines at X of origin.
        /// i.e. origin is the upper middle of the text block.
        /// </summary>
        Center,

        /// <summary>
        /// Center-align text lines at X of origin.
        /// i.e. origin is the upper right corner of the text block.
        /// </summary>
        Right
    }

    /// <summary>
    /// Display (fy line-by-line fade) a collection of info messages.
    /// </summary>
    public class InfoMessages : GraphicalElementBase
    {
        /// <summary>
        /// Text to display
        /// </summary>
        private List<string> messages;

        /// <summary>
        /// Number of text rows to fade in.
        /// </summary>
        private int numRows;

        /// <summary>
        /// Current row being faded.
        /// </summary>
        private int currentRow;

        /// <summary>
        /// The speed of the fade in alpha-change per update.
        /// </summary>
        private float fadeSpeed = 0.1f;

        /// <summary>
        /// Array of alpha values for each text row.
        /// </summary>
        private float[] textAlpha;

        /// <summary>
        /// Origin of text block on the screen.
        /// </summary>
        private Vector2 messagesOrigin;

        /// <summary>
        /// The text font
        /// </summary>
        private SpriteFont font;

        /// <summary>
        /// The text color.
        /// </summary>
        private Color color;

        /// <summary>
        /// The text alignment with respect to the origin.
        /// </summary>
        private InfoMessageAlignment alignment = InfoMessageAlignment.Left;

        /// <summary>
        /// The text scale.
        /// </summary>
        private float scale = 1.0f;

        /// <summary>
        /// Create an info-message display object at the specified location.
        /// Font defaults to the 'game' font. Color defaults to 'white'.
        /// </summary>
        /// <param name="screenManager">The screen manager to use.</param>
        /// <param name="messages">The messages to display.</param>
        /// <param name="origin">Upper right corner where messages are displayed.</param>
        public InfoMessages(
            ScreenManager manager,
            List<string> messages,
            Vector2 origin) : base(manager)
        {
            if (null == messages)
            {
                throw new ArgumentNullException(
                    "messages", 
                    "List of messages cannot be null");
            }

            if (null == origin)
            {
                throw new ArgumentNullException(
                    "origin", 
                    "Text origin cannot be null");
            }

            if (0 == messages.Count)
            {
                throw new ArgumentOutOfRangeException(
                    "messages", 
                    "Message List cannot be empty");
            }

            // Remember messages and origin
            this.messages = messages;
            this.messagesOrigin = origin;

            // Reset text fader state
            this.numRows = messages.Count;
            this.currentRow = 0;
            this.textAlpha = new float[this.numRows];
            for (int i = 0; i < this.numRows; i++)
            {
                this.textAlpha[i] = 0.0f;
            }

            // Initialize a font
            this.font = this.ScreenManager.Fonts["game"];

            // Initialize the color
            Color color = Color.White;
        }

        /// <summary>
        /// Gets or sets the text font.
        /// </summary>
        public SpriteFont Font
        {
            get { return this.font; }
            set { this.font = value; }
        }

        /// <summary>
        /// Gets or sets the text color.
        /// </summary>
        public Color Color
        {
            get { return this.color; }
            set { this.color = value; }
        }

        /// <summary>
        /// Gets or sets the origin of the messages.
        /// </summary>
        public Vector2 MessagesOrigin
        {
            get { return this.messagesOrigin; }
            set { this.messagesOrigin = value; }
        }

        /// <summary>
        /// Gets or sets the alignment of the rows in the text block.
        /// </summary>
        public InfoMessageAlignment Alignment
        {
            get { return this.alignment; }
            set { this.alignment = value; }
        }

        /// <summary>
        /// Gets or sets the text scale.
        /// </summary>
        public float Scale
        {
            get { return this.scale; }
            set { this.scale = value; }
        }

        /// <summary>
        /// Updates the state of the message list display.
        /// </summary>
        /// <param name="gameTime">The current game time.</param>
        public void Update(GameTime gameTime)
        {
            // Fade logic
            if (this.currentRow < this.numRows)
            {
                this.textAlpha[this.currentRow] += this.fadeSpeed;
                if (string.IsNullOrEmpty(this.messages[this.currentRow]) ||
                   (this.textAlpha[this.currentRow] >= 1.0f))
                {
                    this.textAlpha[this.currentRow] = 1.0f;
                    this.currentRow++;
                }
            }
        }

        /// <summary>
        /// Draws the tes message lines onto screen.
        /// </summary>
        /// <param name="gameTime">Current game time.</param>
        public void Draw(GameTime gameTime)
        {            
            Vector2 lineOrigin = new Vector2(0, 0);
            Vector2 linePosition = this.messagesOrigin;
            Vector2 textPosition;
            this.SpriteBatch.Begin();
            for (int i = 0; i < numRows; i++)
            {
                // Current message
                string message = messages[i];

                // Align
                textPosition = linePosition;
                Vector2 textSize = font.MeasureString(message) * this.scale;
                switch (this.alignment)
                {
                    case InfoMessageAlignment.Left:
                        // already aligned
                        break;
                    case InfoMessageAlignment.Center:
                        textPosition.Y -= (font.LineSpacing / 2);
                        textPosition.X -= (textSize.X / 2);
                        break;
                    case InfoMessageAlignment.Right:
                        textPosition.X -= textSize.X;
                        break;
                }

                if (textAlpha[i] > 0.0f)
                {
                    if (string.IsNullOrEmpty(message))
                    {
                        // Spacer
                        linePosition.Y += ((float)font.LineSpacing * 0.5f * this.scale);
                    }
                    else
                    {
                        // Textline
                        color = new Color(
                            Color.White, 
                            MathHelper.SmoothStep(0.0f, 1.0f, textAlpha[i]));
                        this.SpriteBatch.DrawString(
                            font,
                            message,
                            textPosition,
                            color,
                            0,
                            lineOrigin,
                            this.scale,
                            SpriteEffects.None,
                            0);
                        linePosition.Y += ((float)font.LineSpacing * 1.2f * this.scale);
                    }
                }
            }
            this.SpriteBatch.End();
        }
    }
}
