﻿//-----------------------------------------------------------------------------
// ShootingGallery.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;
    using NewGamePhysics.PhysicalElements;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;

    /// <summary>
    /// Represents a collection of moving circular object in 2D space
    /// under gravity influence which can be release (shot down).
    /// </summary>
    public class ShootingGallery : GraphicalElementBase
    {
        /// <summary>
        /// Total number of game targets.
        /// </summary>
        private int numTargets = 0;

        /// <summary>
        /// Number of active (hittable) targets.
        /// </summary>
        private int numActiveTargets = 0;

        /// <summary>
        /// The origin for the gallery in simulation space.
        /// </summary>
        Vector2 simulationOrigin;

        /// <summary>
        /// The width of the gallery
        /// </summary>
        private double width;

        /// <summary>
        /// The height of the gallery
        /// </summary>
        private double height;

        /// <summary>
        /// The origin for mapping simulation space to the screen.
        /// </summary>
        Vector2 screenOrigin;

        /// <summary>
        /// The scale for mapping simulation space to the screen.
        /// </summary>
        double screenScale;

        /// <summary>
        /// The targets.
        /// </summary>
        private ShootingGalleryTarget[] shootingGalleryTargets = null;

        /// <summary>
        /// Target conveyer animation: horizontal speed.
        /// Default: dX = 0.01
        /// </summary>
        private double targetConveyerSpeed = 0.001;

        /// <summary>
        /// Target conveyer animation: horizontal speed.
        /// Default: dX = 0.01
        /// </summary>
        public double TargetConveyerSpeed
        {
            get { return this.targetConveyerSpeed; }
            set { this.targetConveyerSpeed = value; }
        }

        /// <summary>
        /// Gets the total number of game targets.
        /// </summary>
        public int NumTargets
        {
            get { return this.numTargets; }
        }

        /// <summary>
        /// Gets the number of active (hittable) targets.
        /// </summary>
        public int NumActiveTargets
        {
            get { return this.numActiveTargets; }
        }

        /// <summary>
        /// Instantiate an empty shooting gallery.
        /// </summary>
        /// <param name="screenManager">The screen manager to use.</param>
        /// <param name="simulationOrigin">The origin for gallery in simulation space.</param>
        /// <param name="width">The width of the gallery in simulation space.</param>
        /// <param name="height">The height of the gallery in simulation space.</param>
        /// <param name="screenOrigin">The origin for mapping simulation space to the screen.</param>
        /// <param name="screenScale">The scale for mapping simulation space to the screen.</param>
        public ShootingGallery(
            ScreenManager screenManager,
            Vector2 simulationOrigin,
            double width,
            double height,
            Vector2 screenOrigin,
            double screenScale)
            : base(screenManager)
        {
            this.simulationOrigin = simulationOrigin;
            this.width = width;
            this.height = height;
            this.screenOrigin = screenOrigin;
            this.screenScale = screenScale;
        }

        /// <summary>
        /// Reset the shooting gallery.
        /// </summary>
        /// <param name="numTargets">Number of new targets in gallery.</param>
        /// <param name="gravity">Gravity environment for gallery.</param>
        /// <param name="atmosphereDensity">Density of the atmosphere.</param>
        public void Reset(string[] targetTypes, double gravity, double atmosphereDensity)
        {
            if (targetTypes == null || targetTypes.Length == 0)
            {
                throw new ArgumentException(
                    "Can reset only to one or more targets",
                    "targetTypes");
            }

            this.numTargets = targetTypes.Length;

            if (gravity == 0.0)
            {
                throw new ArgumentException(
                    "Gravity must be non-zero.",
                    "gravity");
            }

            // Separation of targets
            double stepX = this.width / this.numTargets;
            if (stepX <= this.height)
            {
                throw new ArgumentException(
                    "Cannot fit that many targets into the width of the gallery without overlap.",
                    "numTargets");
            }

            this.numActiveTargets = this.numTargets;
            this.shootingGalleryTargets = new ShootingGalleryTarget[this.numTargets];
            for (int i = 0; i < this.numTargets; i++)
            {
                this.shootingGalleryTargets[i] = new ShootingGalleryTarget(
                    this.ScreenManager, 
                    (ShootingGalleryTargetType)Enum.Parse(
                        typeof(ShootingGalleryTargetType), 
                        targetTypes[i]),
                    gravity,
                    atmosphereDensity,
                    this.height,
                    this.height,
                    this.screenOrigin,
                    this.screenScale);
                this.shootingGalleryTargets[i].Position = 
                    new Vector2(
                        this.simulationOrigin.X + (float)(i * stepX), 
                        this.simulationOrigin.Y);
            }
        }
        
        /// <summary>
        /// Shoot at targets.
        /// </summary>
        /// <param name="shotCenter">Position of the shot in simulation space.</param>
        /// <param name="bulletSize">Size of the bullet in simulation space.</param>
        /// <returns>Number of targets that were hit.</returns>
        public int ShootAtTargets(Vector2 shotCenter, double bulletSize)
        {
            int numberHits = 0;

            foreach (ShootingGalleryTarget shootingGalleryTarget in this.shootingGalleryTargets)
            {
                if (shootingGalleryTarget.Visible && !shootingGalleryTarget.Animated)
                {
                    if (shootingGalleryTarget.ShootAtTarget(shotCenter, bulletSize))
                    {
                        numberHits++;
                    }
                }
            }

            return numberHits;
        }

        /// <summary>
        /// Update (animate) the shooting gallery. Tracks targets that become invisible
        /// because they were shot off the screen.
        /// </summary>
        /// <param name="gameTime">The game time for updating.</param>
        public void Update(GameTime gameTime)
        {
            foreach (ShootingGalleryTarget shootingGalleryTarget in this.shootingGalleryTargets)
            {
                if (shootingGalleryTarget.Visible)
                {
                    if (shootingGalleryTarget.Animated)
                    {
                        shootingGalleryTarget.Update();
                        if (!shootingGalleryTarget.Visible)
                        {
                            this.numActiveTargets--;
                        }
                    }
                    else
                    {
                        shootingGalleryTarget.Move(this.targetConveyerSpeed);

                        // Wrap horizontal position around
                        Vector2 position = shootingGalleryTarget.Position;
                        if ((position.X - this.simulationOrigin.X) < 0.0)
                        {
                            position.X = this.simulationOrigin.X + (float)this.width;
                            shootingGalleryTarget.Position = position;
                        }
                        else if ((position.X - this.simulationOrigin.X) > (float)this.width)
                        {
                            position.X = this.simulationOrigin.X;
                            shootingGalleryTarget.Position = position;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Draw the shooting gallery.
        /// </summary>
        /// <param name="gameTime">The game time for drawing.</param>
        public void Draw(GameTime gameTime)
        {
            foreach (ShootingGalleryTarget shootingGalleryTarget in this.shootingGalleryTargets)
            {
                shootingGalleryTarget.Draw(gameTime);
            }
        }
    }
}
