﻿//-----------------------------------------------------------------------------
// ShootingGalleryTarget.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Enum specifying the various targets we can shoot down.
    /// </summary>
    public enum ShootingGalleryTargetType
    {
        /// <summary>
        /// Ball of copper.
        /// </summary>
        CopperBall,

        /// <summary>
        /// Ball of silver.
        /// </summary>
        SilverBall,

        /// <summary>
        /// Ball of gold.
        /// </summary>
        GoldBall,
    }

    /// <summary>
    /// Encapsulates a shooting gallery target which can move on a 
    /// horizontal conveyor or fall under the influence of gravity.
    /// </summary>
    public class ShootingGalleryTarget : GraphicalElementBase
    {
        /// <summary>
        /// The physical target simulation.
        /// </summary>
        private CircularObjectSimulation animatedCircularObject;

        /// <summary>
        /// The width of target in simulation space.
        /// </summary>
        private double width;

        /// <summary>
        /// The height of target in simulation space.
        /// </summary>
        private double height;

        /// <summary>
        /// The origin for mapping simulation space to the screen.
        /// </summary>
        Vector2 screenOrigin;

        /// <summary>
        /// The scale for mapping simulation space to the screen.
        /// </summary>
        double screenScale;

        /// <summary>
        /// The destination position on the screen.
        /// </summary>
        Vector2 screenPosition;

        /// <summary>
        /// The texture of the target image.
        /// </summary>
        private Texture2D targetTexture;

        /// <summary>
        /// Flag indicating if the target is visible.
        /// </summary>
        private bool visible;

        /// <summary>
        /// Flag indicating if the target is animated (falling).
        /// </summary>
        private bool animated;

        /// <summary>
        /// The radius of a circular target.
        /// </summary>
        private double radius;

        /// <summary>
        /// The texture-to-display scale of the sprite.
        /// </summary>
        Vector2 spriteScale;

        /// <summary>
        /// The origin of the sprite.
        /// </summary>
        Vector2 spriteOrigin;

        /// <summary>
        /// Creates an instance of the target if the specified type and gravity environment.
        /// Center is set to (0,0) initially, target is visible and not animated.
        /// </summary>
        /// <param name="screenManager">The screen manager to use for drawing.</param>
        /// <param name="type">The type of the target.</param>
        /// <param name="atmosphereDensity">The density of the atmosphere.</param>
        /// <param name="gravity">The gravity of the simulation.</param>
        /// <param name="simulationWidth">The width of the simulated target.</param>
        /// <param name="simulationHeight">The height of the simulated target.</param>
        /// <param name="screenOrigin">The origin for mapping simulation space to the screen.</param>
        /// <param name="screenScale">The scale for mapping simulation space to the screen.</param>
        public ShootingGalleryTarget(
            ScreenManager screenManager, 
            ShootingGalleryTargetType type, 
            double gravity,
            double atmosphereDensity,
            double simulationWidth,
            double simulationHeight,
            Vector2 screenOrigin, 
            double screenScale)
            : base(screenManager)
        {
            this.width = simulationWidth;
            this.height = simulationHeight;
            this.screenOrigin = screenOrigin;
            this.screenScale = screenScale;
            this.radius = Math.Min(width, height) / 2;
            double diameter = 2.0 * this.radius;
            this.screenPosition = new Vector2();
            
            this.visible = true;
            this.animated = false;

            string textureName = string.Empty;
            switch (type)
            {
                case ShootingGalleryTargetType.CopperBall:
                    /// Reference: http://en.wikipedia.org/wiki/Copper (g·cm−3)
                    this.animatedCircularObject =
                        new CircularObjectSimulation(diameter, 8.94, gravity, atmosphereDensity);
                    textureName = "copperball";
                    break;
                case ShootingGalleryTargetType.SilverBall:
                    /// Reference: http://en.wikipedia.org/wiki/Silver (g·cm−3)
                    this.animatedCircularObject =
                        new CircularObjectSimulation(diameter, 10.49, gravity, atmosphereDensity);
                    textureName = "silverball";
                    break;
                case ShootingGalleryTargetType.GoldBall:
                    /// Reference: http://en.wikipedia.org/wiki/Gold (g·cm−3)
                    this.animatedCircularObject =
                        new CircularObjectSimulation(diameter, 19.30, gravity, atmosphereDensity);
                    textureName = "goldball";
                    break;
            }

            // Prepare texture
            if (screenManager.Textures.ContainsKey(textureName))
            {
                this.targetTexture = screenManager.Textures[textureName];
            }
            else
            {
                throw new ArgumentException("Texture with name " + textureName + " not found");
            }

            float spriteScaleX = (float)(this.width * this.screenScale / this.targetTexture.Width);
            float spriteScaleY = (float)(this.height * this.screenScale / this.targetTexture.Height);
            this.spriteScale = new Vector2(spriteScaleX, spriteScaleY);
            this.spriteOrigin = new Vector2();
        }

        /// <summary>
        /// Gets of sets a flag indicating if the target is animated (falling).
        /// </summary>
        public bool Animated
        {
            get { return this.animated; }
            set { this.animated = value; }
        }

        /// <summary>
        /// Gets of sets a flag indicating if the target is visible.
        /// </summary>
        public bool Visible
        {
            get { return this.visible; }
            set { this.visible = value; }
        }

        /// <summary>
        /// Sets or sets the center of the animated target object in simulation space.
        /// </summary>
        public Vector2 Position
        {
            get { return this.animatedCircularObject.GetPosition(); }
            set { this.animatedCircularObject.SetPosition(value); }
        }

        /// <summary>
        /// Move the target by the specified amount if visible.
        /// </summary>
        /// <param name="displacement">The horizontal displacement in simulation space.</param>
        public void Move(double horizontalDisplacement)
        {
            if (this.visible)
            {
                Vector2 position = 
                    this.animatedCircularObject.GetPosition();
                position.X += (float)horizontalDisplacement;
                this.animatedCircularObject.SetPosition(position);
            }
        }

        /// <summary>
        /// Shoot at the target. Sets target to animated if hit.
        /// </summary>
        /// <param name="shotCenter">Position of the shot in simulation space.</param>
        /// <param name="bulletSize">Size of the bullet in simulation space.</param>
        /// <returns>Flag indicating a hit.</returns>
        public bool ShootAtTarget(Vector2 shotCenter, double bulletSize)
        {
            if (this.visible)
            {
                Vector2 targetPosition = 
                    this.animatedCircularObject.GetPosition();

                if (IntersectionTest.CircleInCircle2D(
                    targetPosition,
                    this.radius,
                    shotCenter,
                    bulletSize))
                {
                    this.animated = true;
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Update (animate) a target object if visible and animated.
        /// May make target invisible if it leaves the screen.
        /// </summary>
        public void Update()
        {
            if (this.visible && this.animated)
            {
                // Let target fall
                this.animatedCircularObject.Animate();

                // Check if we left the screen
                UpdateScreenPosition();
                if (this.screenPosition.Y > (this.Viewport.Height + this.radius * this.screenScale))
                {
                    this.visible = false;
                }
            }
        }

        /// <summary>
        /// Draw a target object if visible.
        /// </summary>
        /// <param name="gameTime">The game time for drawing.</param>
        public void Draw(GameTime gameTime)
        {
            if (this.visible)
            {
                // Get fresh screen coordinates
                UpdateScreenPosition();

                // Draw target
                this.SpriteBatch.Begin(SpriteBlendMode.AlphaBlend);
                this.SpriteBatch.Draw(
                    this.targetTexture,
                    this.screenPosition,
                    null,
                    Color.White,
                    0.0f,
                    this.spriteOrigin,
                    this.spriteScale,
                    SpriteEffects.None, 
                    0.0f);  
                this.SpriteBatch.End();
            }
        }

        /// <summary>
        /// Recalculate the screen position of the target.
        /// </summary>
        private void UpdateScreenPosition()
        {
            Vector2 targetPosition =
                this.animatedCircularObject.GetPosition();
            screenPosition.X = (float)(this.screenOrigin.X + this.screenScale * (targetPosition.X - this.width / 2));
            screenPosition.Y = (float)(this.screenOrigin.Y + this.screenScale * (targetPosition.Y - this.height / 2));
        }
    }
}
