﻿
namespace NewGamePhysics.GraphicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.Mathematics;

    public class TexturedSphere
    {
        VertexBuffer vertexbuffer;  
        IndexBuffer indexbuffer;  
        BasicEffect effect;  
        VertexDeclaration declaration;  

        int strideSize;  
        int nvertices;  
        int nfaces;  

        Matrix sphereRollingMatrix = Matrix.Identity;
 
        /// <summary>  
        /// Generates a simple textured sphere  
        /// </summary>  
        /// <param name="device">The graphics device context to create the 3D sphere in.</param>  
        /// <param name="Stacks">The number of stacks in the sphere.</param>  
        /// <param name="Slices">The number of slices in the spehre.</param>  
        /// <param name="Radius">The radius of the sphere</param>  
        /// <param name="texture">The texture of the sphere.</param>
        public TexturedSphere(GraphicsDevice device, int Stacks, int Slices, float Radius, Texture2D texture)  
        {  
            // calculates the resulting number of vertices and indices  
            nvertices = (Stacks + 1) * (Slices + 1);  
            VertexPositionNormalTexture[] vertices = new VertexPositionNormalTexture[nvertices];  
 
            int numIndices = (3 * Stacks * (Slices + 1)) * 2;  
            int[] indices = new int[numIndices];
            nfaces = numIndices/3;
              
            float StackAngle = MathHelper.Pi / (float)Stacks;  
            float SliceAngle = (float)(Math.PI * 2.0) / (float)Slices;  
 
            int vertexindex = 0;  

            // Generate the group of Stacks for the sphere  
            int vertexcount = 0;  
            int indexcount = 0;  
 
            for (int stack = 0; stack < (Stacks+1); stack++)  
            {  
 
                float r = (float)Math.Sin((float)stack * StackAngle);  
                float y = (float)Math.Cos((float)stack * StackAngle);

                double lat = ((double)stack / (double)Stacks - 0.5) * 180.0;
                double v = EquirectangularProjection.latToY(lat);

                // Generate the group of segments for the current Stack  
                for (int slice = 0; slice < (Slices+1); slice++)  
                {  
                    float x = r * (float)Math.Sin((float)slice * SliceAngle);  
                    float z = r * (float)Math.Cos((float)slice * SliceAngle);  
                    vertices[vertexcount].Position = new Vector3(x * Radius, y * Radius, z * Radius);   
                    vertices[vertexcount].Normal = Vector3.Normalize(new Vector3(x, y, z));

                    double lon = ((double)slice / (double)Slices) * 360.0;
                    double u = 1.0 - EquirectangularProjection.lonToX(lon);
 
                    vertices[vertexcount].TextureCoordinate = new Vector2((float)u, (float)v);

                    vertexcount++;  
                    if (!(stack == (Stacks - 1)))  
                    {  
                        indices[indexcount] = vertexindex + (Slices + 1);  
                        indexcount++;  
                        indices[indexcount] = vertexindex + 1;  
                        indexcount++;  
                        indices[indexcount] = vertexindex;  
                        indexcount++;  
                        indices[indexcount] = vertexindex + (Slices);  
                        indexcount++;  
                        indices[indexcount] = vertexindex + (Slices + 1);  
                        indexcount++;  
                        indices[indexcount] = vertexindex;  
                        indexcount++;  
                        vertexindex++;  
                    }  
                }  
            }  
 
            vertexbuffer = new VertexBuffer(device, typeof(VertexPositionNormalTexture), nvertices, BufferUsage.None);  
            vertexbuffer.SetData(vertices, 0, vertices.Length);  
            indexbuffer = new IndexBuffer(device ,  typeof(int), numIndices, BufferUsage.None);  
            indexbuffer.SetData(indices, 0, indices.Length);  
 
            this.effect = new BasicEffect(device, null);
            this.effect.Texture = texture;
            this.effect.Alpha = 1.0f;
            this.effect.DiffuseColor = new Vector3(1.0f, 1.0f, 1.0f);
            this.effect.SpecularColor = new Vector3(0.25f, 0.25f, 0.25f);
            this.effect.SpecularPower = 5.0f;
            this.effect.AmbientLightColor = new Vector3(0.75f, 0.75f, 0.75f);
            this.effect.TextureEnabled = true;

            declaration = new VertexDeclaration(device, VertexPositionNormalTexture.VertexElements);  
            strideSize = VertexPositionNormalTexture.SizeInBytes;  
        }

        /// <summary>
        /// Render textured sphere.
        /// </summary>
        /// <param name="device"></param>
        /// <param name="view"></param>
        /// <param name="projection"></param>
        /// <param name="sphereRollingMatrix"></param>
        public void Render(GraphicsDevice device, Matrix view, Matrix projection, Matrix sphereRollingMatrix, float alpha)
        {
            device.Indices = indexbuffer;
            device.VertexDeclaration = declaration;
            device.Vertices[0].SetSource(vertexbuffer, 0, strideSize);
            this.effect.World = sphereRollingMatrix;
            this.effect.Projection = projection;
            this.effect.View = view;
            this.effect.Alpha = alpha;
            this.effect.Begin();

            foreach (EffectPass effectPass in this.effect.CurrentTechnique.Passes)
            {
                effectPass.Begin();

                device.DrawIndexedPrimitives(PrimitiveType.TriangleList, 0, 0, nvertices, 0, nfaces);

                effectPass.End();
            }
            this.effect.End();
        }  
    }
}
