﻿//-----------------------------------------------------------------------------
// Bicubic.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;

    /// <summary>
    /// Bicubic value interpolator.
    /// Reference: 
    /// http://mrl.nyu.edu/~perlin/java/Bicubic.html
    /// </summary>
    public class Bicubic
    {
        /// <summary>
        /// Catmull-Rom basis matrix
        /// </summary>
        private double[,] M = {             
            {-0.5,  1.5, -1.5, 0.5},
            { 1  , -2.5,  2  ,-0.5},
            {-0.5,  0  ,  0.5, 0  },
            { 0  ,  1  ,  0  , 0  }};

        /// <summary>
        /// The coefficient matrix.
        /// </summary>
        private double[,] C = new double[4, 4];

        /// <summary>
        /// Calculates bicubic coefficients from 16 grid values.
        /// </summary>
        /// <param name="G">Values at at [-1,0,1,2]x[-1,0,1,2].</param>
        public Bicubic(double[,] G)
        {
            if ((G == null) && (G.Length != 16))
            {
                throw new ArgumentOutOfRangeException("G");
            }

            double[,] T = new double[4, 4];

            // T = G * MT
            for (int i = 0; i < 4; i++)
            {
                for (int j = 0; j < 4; j++)
                {
                    for (int k = 0; k < 4; k++)
                    {
                        T[i, j] += G[i, k] * M[j, k];
                    }
                }
            }

            // C = M * T
            for (int i = 0; i < 4; i++)
            {
                for (int j = 0; j < 4; j++)
                {
                    for (int k = 0; k < 4; k++)
                    {
                        C[i, j] += M[i, k] * T[k, j];
                    }
                }
            }
        }

        /// <summary>
        /// Calculate bicubic interpolate on a unit square.
        /// </summary>
        /// <param name="x">The x coordinate.</param>
        /// <param name="y">The y coordinate.</param>
        /// <returns>The interpolated value at x,y</returns>
        public double Calc(double x, double y)
        {
            if ((x < 0.0) || (x > 1.0))
            {
                throw new ArgumentOutOfRangeException("x");
            }

            if ((y < 0.0) || (y > 1.0))
            {
                throw new ArgumentOutOfRangeException("y");
            }

            return x * (x * (x * (y * (y * (y * C[0, 0] + C[0, 1]) + C[0, 2]) + C[0, 3])
                               + (y * (y * (y * C[1, 0] + C[1, 1]) + C[1, 2]) + C[1, 3]))
                               + (y * (y * (y * C[2, 0] + C[2, 1]) + C[2, 2]) + C[2, 3]))
                               + (y * (y * (y * C[3, 0] + C[3, 1]) + C[3, 2]) + C[3, 3]);
        }
    }
}
