﻿//-----------------------------------------------------------------------------
// Gamma.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;

    /// <summary>
    /// Special functions: Gamma functions.
    /// </summary>
    public class Gamma
    {
        /// <summary>
        /// Calculates the logarithm of the gamma function ln(G(x)).
        /// Evaluated iteratively.
        /// Reference: 
        /// M. Pike, D. Hill, Algorithm 291: Logarithm of the Gamma Function,
        /// Communications of the ACM, Vol. 9, #9, 09/1966, page 684.
        /// See also:
        /// http://mathworld.wolfram.com/LogGammaFunction.html
        /// </summary>
        /// <param name="x">The argument of the log gamma function. Value range >0.</param>
        /// <returns>The logarithm of the gamma function at x.</returns>
        public static double LogFunction(double x)
        {
            if (x <= 0)
            {
                throw new ArgumentOutOfRangeException("x");
            }

            double f;
            double y;
            double z;

            y = x;
            if (x < 7.0)
            {
                f = 1.0;
                z = y;
                while (z < 7.0)
                {
                    f *= z;
                    z += 1.0;
                }

                y = z;
                f = -Math.Log(f);
            }
            else
            {
                f = 0.0;
            }

            z = 1.0 / (y * y);

            double v =
                f + (y - 0.5) * Math.Log(y) - y + 0.918938533204673 +
                (((-0.000595238095238 * z + 0.000793650793651) * z - 0.002777777777778) * z + 0.083333333333333) / y;

            return v;
        }
    }
}
