﻿//-----------------------------------------------------------------------------
// Legendre.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;

    /// <summary>
    /// Special functions: Legendre functions.
    /// Provides methods to calculate Legendre polynomials, 
    /// associated functions, sums and coefficient sets.
    /// </summary>
    public static class Legendre
    {
        /// <summary>
        /// Temporary variable cache.
        /// </summary>
        private static double[] f1;

        /// <summary>
        /// Temporary variable cache.
        /// </summary>
        private static double[] f2;

        /// <summary>
        /// Temporary variable cache.
        /// </summary>
        private static double[] sqr;

        /// <summary>
        /// Last harmonic degree of temporary variables.
        /// </summary>
        private static int lmax_old = -1;

        /// <summary>
        /// Calculation of the value of the Legendre polynomial P_n.
        /// Fast inline calculation up to order n=19, then iterative.
        /// </summary>
        /// <param name="n">The degree of the polynomial. Valid range >=0.</param>
        /// <param name="x">The argument to evaluate. Valid range is [-1,1].</param>
        /// <returns>The value of the Legendre polynomial Pn at x.</returns>
        public static double Polynomial(int n, double x)
        {
            if (n < 0)
            {
                throw new ArgumentOutOfRangeException("n");
            }

            if ((x < -1.0) || (x > 1.0))
            {
                throw new ArgumentOutOfRangeException("x");
            }

            double x2, x4, x6, x8, x10, x12, x14, x16, x18;

            switch (n)
            {
                case 0:
                    return (1.0);
                case 1:
                    return (x);
                case 2:
                    x2 = x * x;
                    return ((3.0 * x2 - 1.0) * 0.5);
                case 3:
                    x2 = x * x;
                    return (x * (5.0 * x2 - 3.0) * 0.5);
                case 4:
                    x2 = x * x;
                    x4 = x2 * x2;
                    return ((35.0 * x4 - 30.0 * x2 + 3.0) * 0.125);
                case 5:
                    x2 = x * x;
                    x4 = x2 * x2;
                    return (x * (63.0 * x4 - 70.0 * x2 + 15.0) * 0.125);
                case 6:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    return ((231.0 * x6 - 315.0 * x4 + 105.0 * x2 - 5.0) * 0.0625);
                case 7:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    return (x * (429.0 * x6 - 693.0 * x4 + 315.0 * x2 - 35.0) * 0.0625);
                case 8:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    return ((6435.0 * x8 - 12012.0 * x6 + 6930.0 * x4 - 1260.0 * x2 + 35.0) * 0.0078125);
                case 9:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    return (x * (12155.0 * x8 - 25740.0 * x6 + 18018.0 * x4 - 4620.0 * x2 + 315.0) * 0.0078125);
                case 10:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    return ((46189.0 * x10 - 109395.0 * x8 + 90090.0 * x6 - 30030.0 * x4 + 3465.0 * x2 - 63.0) * 0.00390625);
                case 11:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    return (x * (88179.0 * x10 - 230945.0 * x8 + 218790.0 * x6 - 90090.0 * x4 + 15015.0 * x2 - 693.0) * 0.00390625);
                case 12:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    x12 = x10 * x2;
                    return ((676039.0 * x12 - 1939938.0 * x10 + 2078505.0 * x8 - 1021020.0 * x6 + 225225.0 * x4 - 18018.0 * x2 + 231.0) * 0.0009765625);
                case 13:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    x12 = x10 * x2;
                    return (x * (1300075.0 * x12 - 4056234.0 * x10 + 4849845.0 * x8 - 2771340.0 * x6 + 765765.0 * x4 - 90090.0 * x2 + 3003.0) * 0.0009765625);
                case 14:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    x12 = x10 * x2;
                    x14 = x12 * x2;
                    return ((5014575.0 * x14 - 16900975.0 * x12 + 22309287.0 * x10 - 14549535.0 * x8 + 4849845.0 * x6 - 765765.0 * x4 + 45045.0 * x2 - 429.0) * 0.00048828125);
                case 15:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    x12 = x10 * x2;
                    x14 = x12 * x2;
                    return (x * (9694845.0 * x14 - 35102025.0 * x12 + 50702925.0 * x10 - 37182145.0 * x8 + 14549535.0 * x6 - 2909907.0 * x4 + 255255.0 * x2 - 6435.0) * 0.00048828125);
                case 16:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    x12 = x10 * x2;
                    x14 = x12 * x2;
                    x16 = x14 * x2;
                    return ((300540195.0 * x16 - 1163381400.0 * x14 + 1825305300.0 * x12 - 1487285800.0 * x10 + 669278610.0 * x8 - 162954792.0 * x6 + 19399380.0 * x4 - 875160.0 * x2 + 6435.0) * 3.0517578125e-05);
                case 17:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    x12 = x10 * x2;
                    x14 = x12 * x2;
                    x16 = x14 * x2;
                    return (x * (583401555.0 * x16 - 2404321560.0 * x14 + 4071834900.0 * x12 - 3650610600.0 * x10 + 1859107250.0 * x8 - 535422888.0 * x6 + 81477396.0 * x4 - 5542680.0 * x2 + 109395.0) * 3.0517578125e-05);
                case 18:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    x12 = x10 * x2;
                    x14 = x12 * x2;
                    x16 = x14 * x2;
                    x18 = x16 * x2;
                    return ((2268783825.0 * x18 - 9917826435.0 * x16 + 18032411700.0 * x14 - 17644617900.0 * x12 + 10039179150.0 * x10 - 3346393050.0 * x8 + 624660036.0 * x6 - 58198140.0 * x4 + 2078505.0 * x2 - 12155.0) * 1.52587890625e-05);
                case 19:
                    x2 = x * x;
                    x4 = x2 * x2;
                    x6 = x4 * x2;
                    x8 = x6 * x2;
                    x10 = x8 * x2;
                    x12 = x10 * x2;
                    x14 = x12 * x2;
                    x16 = x14 * x2;
                    x18 = x16 * x2;
                    return (x * (4418157975.0 * x18 - 20419054425.0 * x16 + 39671305740.0 * x14 - 42075627300.0 * x12 + 26466926850.0 * x10 - 10039179150.0 * x8 + 2230928700.0 * x6 - 267711444.0 * x4 + 14549535.0 * x2 - 230945.0) * 1.52587890625e-05);
                default:
                    double result = 1;
                    double a = 1;
                    double b = x;
                    for (int i = 2; i <= n; i++)
                    {
                        result = ((double)(2 * i - 1) * x * b - (double)(i - 1) * a) / (double)i;
                        a = b;
                        b = result;
                    }
                    return result;
            }
        }

        /// <summary>
        /// Return the associated Legendre function P_l^m(x). 
        /// Evaluated iteratively. Default funtion to use.
        /// References:
        /// M. Abramovitz, I. A. Stegun (ed.), Handbook of Mathematical Functions:
        /// Dover Publications, Sec. 8, pp. 331-341.
        /// Matpak, http://users.physik.tu-muenchen.de/gammel/matpack/html/matpack_contents.html#library
        /// </summary>
        /// <param name="l">The first order. Value range >=0.</param>
        /// <param name="m">The second order. Value range >=0 and  m&lt;=n </param>
        /// <param name="x">The argument to evaluate. Valid range is [-1,1].</param>
        /// <returns>The value of the associated Legendre function P_l^m at x.</returns>
        public static double AssociatedFunction(int l, int m, double x)
        {
            if (l < 0)
            {
                throw new ArgumentOutOfRangeException("l");
            }

            if ((m < 0) || (m > l))
            {
                throw new ArgumentOutOfRangeException("m");
            }

            if ((x < -1.0) || (x > 1.0))
            {
                throw new ArgumentOutOfRangeException("x");
            }

            double p = 1.0;
            if (m > 0)
            {
                double h = Math.Sqrt((1.0 - x) * (1.0 + x));
                double f = 1.0;
                for (int i = 1; i <= m; i++)
                {
                    p *= (-f * h);
                    f += 2.0;
                }
            }
            if (l == m)
            {
                return p;
            }
            else
            {
                double pp = x * (double)(2 * m + 1) * p;
                if (l == (m + 1))
                {
                    return pp;
                }
                else
                {
                    double pll = 0.0;
                    for (int ll = m + 2; ll <= l; ll++)
                    {
                        pll = (x * (double)(2 * ll - 1) * pp - (double)(ll + m - 1) * p) / (double)(ll - m);
                        p = pp;
                        pp = pll;
                    }
                    return pll;
                }
            }
        }

        /// <summary>
        /// Return the associated Legendre function P_l^m(x). 
        /// Evaluated iteratively. 
        /// NOTE: Seems to have sign issues.
        /// References:
        /// M. Abramovitz, I. A. Stegun (ed.), Handbook of Mathematical Functions:
        /// Dover Publications, Sec. 8, pp. 331-341.
        /// Shodor, http://www.shodor.org/refdesk/Resources/Libraries/AssocLegendre/download.php
        /// </summary>
        /// <param name="l">The first order. Value range >=0.</param>
        /// <param name="m">The second order. Value range >=0 and  m&lt;=n </param>
        /// <param name="x">The argument to evaluate. Valid range is [-1,1].</param>
        /// <returns>The value of the associated Legendre function P_l^m at x.</returns>
        public static double AssociatedFunctionShodor(int l, int m, double x)
        {
            if (l < 0)
            {
                throw new ArgumentOutOfRangeException("l");
            }

            if ((m < 0) || (m > l))
            {
                throw new ArgumentOutOfRangeException("m");
            }

            if ((x < -1.0) || (x > 1.0))
            {
                throw new ArgumentOutOfRangeException("x");
            }

            double last, current, next;
            if (l < m)
            {
                current = 0.0;
            }
            else if ((l == m) && (m == 0))
            {
                current = 1.0;
            }
            else
            {
                last = 0.0;
                if (m == 0)
                {
                    current = 1.0;
                }
                else
                {
                    current =
                        (double)Factorial.CalcDouble((int)(2 * m - 1)) *
                        Math.Pow((1.0 - x * x), 0.5 * (double)m);
                }
                if (l != m)
                {
                    for (int k = m; k < l; k++)
                    {
                        next =
                            ((double)(2 * k + 1) * x * current -
                            (double)(k + m) * last) / (double)(k + 1 - m);
                        last = current;
                        current = next;
                    }
                }
            }
            return current;
        }

        /// <summary>
        /// Return the associated Legendre function P_l^m(x). 
        /// Evaluated iteratively. 
        /// References:
        /// M. Abramovitz, I. A. Stegun (ed.), Handbook of Mathematical Functions:
        /// Dover Publications, Sec. 8, pp. 331-341.
        /// The Gnu Scientific Library, http://www.gnu.org.
        /// </summary>
        /// <param name="l">The first order. Value range >=0.</param>
        /// <param name="m">The second order. Value range >=0 and  m&lt;=n </param>
        /// <param name="x">The argument to evaluate. Valid range is [-1,1].</param>
        /// <returns>The value of the associated Legendre function P_l^m at x.</returns>
        public static double AssociatedFunctionGsl(int l, int m, double x)
        {
            if (l < 0)
            {
                throw new ArgumentOutOfRangeException("l");
            }

            if ((m < 0) || (m > l))
            {
                throw new ArgumentOutOfRangeException("m");
            }

            if ((x < -1.0) || (x > 1.0))
            {
                throw new ArgumentOutOfRangeException("x");
            }

            if (m == 0)
            {
                return Polynomial(l, x);
            }
            else
            {
                double mm = 1.0;
                if (m > 0)
                {
                    double r = Math.Sqrt(1.0 - x) * Math.Sqrt(1.0 + x);
                    double f = 1.0;
                    for (int i = 1; i <= m; i++)
                    {
                        mm *= (-f * r);
                        f += 2.0;
                    }
                }

                if (l == m)
                {
                    return mm;
                }

                double mpnm = (double)(2 * m + 1) * x * mm;
                if (l == (m + 1))
                {
                    return mpnm;
                }

                double nm2m = mm;
                double nmnm = mpnm;
                double nm = 0.0;
                for (int j = m + 2; j <= l; j++)
                {
                    nm = ((double)(2 * j - 1) * x * nmnm - (double)(j + m - 1) * nm2m) / (double)(j - m);
                    nm2m = nmnm;
                    nmnm = nm;
                }

                return nm;
            }
        }


        /// <summary>
        /// Calculate all normalized associated Legendre functions up to a maximum
        /// order lmax. Normalization is done by geophysical convention. Excludes the
        /// Condon-Shortley phase. Maximum degree possible is 2800.
        /// Reference: Holmes and Featherstone 2002, J. Geodesy, 76, pp. 279-299.
        /// Based on Fortran95 sources of SHTOOLS (available at 
        /// http://www.ipgp.jussieu.fr/~wieczor/SHTOOLS/SHTOOLS.html) by Mark Wieczorek.
        /// </summary>
        /// <param name="lmax">The maximum harmonic degree to calculate.</param>
        /// <param name="z">The value to evaluate, cos(colatitude) or sin(latitude).</param>
        /// <param name="value">
        /// The associated Legendre polynomials evaluated at z up to lmax.
        /// Vector is allocated and upon return contains (lmax+1)*(lmax+2)/2+1 values
        /// in a skewed 2D matrix with the first index coresponding to l and the second to m.
        /// </param>
        /// <param name="derivative">
        /// A vector of all first derivatives of the Legendre polynomials evaluated at z.
        /// Vector is allocated and upon return contains (lmax+1)*(lmax+2)/2+1 values
        /// in a skewed 2D matrix with the first index coresponding to l and the second to m.
        /// </param>
        public static void NormalizedAssociatedFunctionAndDerivative(int lmax, double z, out double[][] value, out double[][] derivative)
        {
            // If set to -1.0, applies phase factor -1^m. No factor otherwise.
            // The Condon-Shortley phase is usually not included in geodesy.
            double phase = 1.0;

            // If set to 1, uses complex normalization. Real otherwise.
            int cnorm = 0;

            // Variables used in calculation
            double[] dp;
            double[] p;
            double pm1;
            double pm2;
            double pmm;
            double plm;
            double rescalem;
            double u;
            double u2;
            double scalef = 1.0e-280;
            int k;
            int kstart;
            int m;
            int mm;
            int l;
            int ll;
            int sqrdim;
            int sdim;

            // Check arguments
            if ((lmax < 0) || (lmax >2800))
            {
                throw new ArgumentOutOfRangeException("lmax");
            }

            if (Math.Abs(z) >= 1.0)
            {
                throw new ArgumentOutOfRangeException("z");
            }

            // Create output arrays
            sdim = (lmax + 1) * (lmax + 2) / 2;
            p = new double[sdim + 1];
            dp = new double[sdim + 1];

            // Check if we need to precompute
            if (lmax != lmax_old)
            {
                lmax_old = lmax;

                // Precompute square root of integers
                sqrdim = 2 * lmax + 1;
                sqr = new double[sqrdim + 1];
                for (l = 1; l <= sqrdim; l++)
                {
                    sqr[l] = Math.Sqrt((double)l);
                }

                // Precompute multiplication factors used in recursion
                f1 = new double[sdim + 1];
                f2 = new double[sdim + 1];
                k = 3;
                for (l = 2; l <= lmax; l++)
                {
                    k++;
                    ll = 2 * l;
                    f1[k] = sqr[ll - 1] * sqr[ll + 1] / (double)l;
                    f2[k] = (double)(l - 1) * sqr[ll + 1] / sqr[ll - 3] / (double)l;
                    for (m = 1; m <= (l - 2); m++)
                    {
                        k++;
                        f1[k] = sqr[ll + 1] * sqr[ll - 1] / sqr[l + m] / sqr[l - m];
                        f2[k] = sqr[ll + 1] * sqr[l - m - 1] * sqr[l + m - 1] / sqr[ll - 3] / sqr[l + m] / sqr[l - m];
                    }
                    k += 2;
                }
            }

            // Calculate unscaled P_l0
            pm2 = 1.0;
            p[1] = 1.0;
            dp[1] = 0.0;

            if (lmax > 0)
            {

                pm1 = sqr[3] * z;
                p[2] = pm1;
                dp[2] = sqr[3];

                u = Math.Sqrt((1.0 - z) * (1.0 + z));
                u2 = u * u;

                k = 2;
                for (l = 2; l <= lmax; l++)
                {
                    k = k + l;
                    ll = 2 * l;
                    plm = f1[k] * z * pm1 - f2[k] * pm2;
                    p[k] = plm;
                    dp[k] = (double)l * (sqr[ll + 1] / sqr[ll - 1] * pm1 - z * plm) / u2;
                    pm2 = pm1;
                    pm1 = plm;
                }

                // Calculate P_mm, P_m1m and P_lm
                if (cnorm == 1)
                {
                    pmm = scalef;
                }
                else
                {
                    pmm = sqr[2] * scalef;
                }

                rescalem = 1.0 / scalef;
                kstart = 1;
                for (m = 1; m <= (lmax - 1); m++)
                {
                    mm = 2 * m;
                    rescalem = rescalem * u;

                    // Calculate P_mm
                    kstart = kstart + m + 1;
                    pmm = phase * pmm * sqr[mm + 1] / sqr[mm];
                    p[kstart] = pmm * rescalem;
                    dp[kstart] = -1.0 * (double)m * z * p[kstart] / u2;
                    pm2 = pmm;

                    // Calculate P_m1m
                    k = kstart + m + 1;
                    pm1 = z * sqr[mm + 3] * pmm;
                    p[k] = pm1 * rescalem;
                    dp[k] = (sqr[mm + 3] * p[k - m - 1] - z * (double)(m + 1) * p[k]) / u2;

                    // Calculate P_lm
                    for (l = m + 2; l <= lmax; l++)
                    {
                        k += l;
                        ll = 2 * l;
                        plm = z * f1[k] * pm1 - f2[k] * pm2;
                        p[k] = plm * rescalem;
                        dp[k] = (sqr[ll + 1] * sqr[l - m] * sqr[l + m] / sqr[ll - 1] * p[k - l] - z * (double)l * p[k]) / u2;
                        pm2 = pm1;
                        pm1 = plm;
                    }
                }

                // Calculate P_lmaxlmax
                rescalem = rescalem * u;
                kstart = kstart + m + 1;
                pmm = phase * pmm * sqr[2 * lmax + 1] / sqr[2 * lmax];
                p[kstart] = pmm * rescalem;
                dp[kstart] = -1.0 * (double)lmax * z * p[kstart] / u2;
            }

            // Remap to output arrays
            // while allocating them
            int i = 0;
            value = new double[lmax + 1][];
            derivative = new double[lmax + 1][];
            for (l = 0; l <= lmax; l++)
            {
                value[l] = new double[l + 1];
                derivative[l] = new double[l + 1];
                for (m = 0; m <= l; m++)
                {
                    i++;
                    value[l][m] = p[i];
                    derivative[l][m] = dp[i];
                }
            }
        }

        /// <summary>
        /// Return the spherical associated Legendre function P_l^m(theta). 
        /// Evaluated iteratively. 
        /// Reference:
        /// M. Abramovitz, I. A. Stegun (ed.), Handbook of Mathematical Functions:
        /// Dover Publications, Sec. 8, pp. 331-341.
        /// The Gnu Scientific Library, http://www.gnu.org.
        /// </summary>
        /// <param name="l">The first order. Value range >=0.</param>
        /// <param name="m">The second order. Value range >=0 and  m&lt;=n </param>
        /// <param name="theta">The radian angle argument to evaluate.</param>
        /// <returns>The value of the spherical associated Legendre function P_l^m at theta.</returns>
        public static double SphericalAssociatedFunction(int l, int m, double theta)
        {
            if (l < 0)
            {
                throw new ArgumentOutOfRangeException("l");
            }

            if ((m < 0) || (m > l))
            {
                throw new ArgumentOutOfRangeException("m");
            }

            double x = Math.Cos(theta);

            if (m == 0)
            {
                double p = Polynomial(l, x);
                double f = Math.Sqrt((double)(2 * l + 1) / (4.0 * Math.PI));
                p *= f;
                return p;
            }
            else if ((x == -1.0) || (x == 1.0))
            {
                return 0.0;
            }
            else
            {
                double sgn = ((m % 2) == 1) ? -1.0 : 1.0;
                double ympnmf = x * Math.Sqrt((double)(2 * m + 3));
                double lnc = Math.Log(1.0 - x * x);
                double lnp = Gamma.LogFunction((double)m + 0.5) - Gamma.LogFunction((double)m);
                double lnpre = -0.25 * Math.Log(Math.PI) + 0.5 * (lnp + (double)m * lnc);
                double sr = Math.Sqrt((2.0 + 1.0 / (double)m) / (4.0 * Math.PI));
                double ymm = sgn * sr * Math.Exp(lnpre);
                double ympnm = ympnmf * ymm;

                if (l == m)
                {
                    return ymm;
                }
                else if (l == (m + 1))
                {
                    return ympnm;
                }
                else
                {
                    double ynm = 0.0;
                    for (int i = m + 2; i <= l; i++)
                    {
                        double r1 = (double)(i - m) / (double)(i + m);
                        double r2 = (double)(i - m - 1) / (double)(i + m - 1);
                        double f1 = Math.Sqrt(r1 * (double)(2 * i + 1) * (double)(2 * i - 1));
                        double f2 = Math.Sqrt(r1 * r2 * (double)(2 * i + 1) / (double)(2 * i - 3));
                        ynm = (x * ympnm * f1 - (double)(i + m - 1) * ymm * f2) / (double)(i - m);
                        ymm = ympnm;
                        ympnm = ynm;
                    }
                    return ynm;
                }
            }
        }

        /// <summary>
        /// Summation of Legendre polynomials using Clenshaw’s recurrence formula
        /// c[0]*P0(x) + c[1]*P1(x) + ... + c[N]*PN(x)
        /// </summary>
        /// <param name="n">The degree of the polynomial. n>=0</param>
        /// <param name="x">The argument to evaluate.</param>
        /// <param name="c">The coefficients. Requires a minimum of n+1 coefficients.</param>
        /// <returns>
        /// The value of the Legendre polynomial at x.
        /// </returns>
        public static double Sum(
            int n,
            double x,
            double[] c
            )
        {
            if (n < 0)
            {
                throw new ArgumentOutOfRangeException("n");
            }

            if ((n + 1) > c.Length)
            {
                throw new ArgumentOutOfRangeException("c", "Not enough coefficients");
            }

            if ((x < -1.0) || (x > 1.0))
            {
                throw new ArgumentOutOfRangeException("x");
            }

            double result = 0;
            double b1 = 0;
            double b2 = 0;

            b1 = 0;
            b2 = 0;
            for (int i = n; i >= 0; i--)
            {
                result = (2 * i + 1) * x * b1 / (i + 1) - (i + 1) * b2 / (i + 2) + c[i];
                b2 = b1;
                b1 = result;
            }

            return result;
        }

        /// <summary>
        /// Calculates the coeffcients c[n] of the legendre
        /// polynomial in the representation of Pn as 
        /// c[0] + c[1]*x + ... + c[n]*x^n
        /// </summary>
        /// <param name="n">Polynomial degree.</param>
        /// <result>Array of n+1 calculated coefficients.</result>
        public static double[] Coefficients(int n)
        {
            if (n < 0)
            {
                throw new ArgumentOutOfRangeException("n");
            }

            int i;
            double[] c = new double[n + 1];
            for (i = 0; i <= n; i++)
            {
                c[i] = 0;
            }

            c[n] = 1;
            for (i = 1; i <= n; i++)
            {
                c[n] = c[n] * (n + i) / 2 / i;
            }

            for (i = 0; i <= n / 2 - 1; i++)
            {
                c[n - 2 * (i + 1)] = -(c[n - 2 * i] * (n - 2 * i) * (n - 2 * i - 1) / 2 / (i + 1) / (2 * (n - i) - 1));
            }
            return c;
        }
    }
}
