//-----------------------------------------------------------------------------
// VectorN.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Mathematics
{
    using System;

    /// <summary>
    /// The VectorN class stores "a point in n-dimensional space" -  
    /// a sequence of real numbers - and defines operators for them.
    /// Based on public code from Vit Buchta, June 2007.
    /// </summary>
    public class VectorN
    {
        /// <summary>
        /// Size of vector.
        /// </summary>
        int n;

        /// <summary>
        /// Scalar elements of vector.
        /// </summary>
        double [] x;

        /// <summary>
        /// A constructor. It defines a number of vector components
        /// and creates an real array with n elements.
        /// </summary>
        /// <param name="n">Number of vector components</param>
        public VectorN(int n)
        {
            if (n < 0)
            {
                throw new ArgumentOutOfRangeException("n","Size must be greater than zero.");
            }

            this.n = n;
            this.x = new double[n];
        }

        /// <summary>
        /// A constructor which copies a given vector into itself
        /// </summary>
        /// <param name="a"> A variable of VectorN type</param>
        public VectorN(VectorN a)
        {
            this.n = a.n;
            this.x = new double[n];
            for (int i = 0; i < this.n; i++)
            {
                this.x[i] = a.x[i];
            }
        }

        /// <summary>
        /// Returns a number of vector components
        /// </summary>
        public int N
        {
            get
            {
                return this.n;
            }
        }

        /// <summary>
        /// An indexer enabling to treat an object of VectorN type as an array
        /// </summary>
        /// <param name="i"> Zero based index</param>
        /// <returns></returns>
        public double this[int i]
        {
            get
            {
                if (i < 0 || i >= this.n)
                {
                    throw new IndexOutOfRangeException("Index out of range.");
                }

                return this.x[i]; 
            }

            set
            {
                if (i < 0 || i >= this.n)
                {
                    throw new IndexOutOfRangeException("Index out of range.");
                }

                this.x[i] = value;
            }
        }

        /// <summary>
        /// Copies itself into another object of VectorN type
        /// </summary>
        /// <returns>An object of VectorN type containing the copy of this object</returns>
        public VectorN Copy()
        {            
            VectorN r = new VectorN(this.n);
            for (int i = 0; i < this.n; i++)
            {
                r.x[i] = this.x[i];
            }

            return r;
        }

        /// <summary>
        /// Absolute value. A method calculating the Euclidean norm
        /// of the vector.
        /// </summary>
        /// <returns> The Eucidean norm of the vector</returns>
        public double Abs()
        {
            double r = 0;
            for (int i = 0; i < this.n; i++)
            {
                double v = this.x[i];
                r += v * v;
            }

            r = Math.Sqrt(r);
            return r;
        }

        /// <summary>
        /// Addition of two vectors
        /// </summary>
        /// <param name="a"> First vector</param>
        /// <param name="b"> Second vector</param>
        /// <returns> A vector created as an addition of vectors a and b</returns>
        public static VectorN operator +(VectorN a, VectorN b)
        {
            if (a.n != b.n)
            {
                throw new ArgumentException("Vectors must be of same size");
            }

            VectorN r = new VectorN(a.n);
            for (int i = 0; i < a.n; i++)
            {
                r.x[i] = a.x[i] + b.x[i];
            }

            return r;
        }

        /// <summary>
        /// VectorN substraction
        /// </summary>
        /// <param name="a"> First vector</param>
        /// <param name="b"> Second vector</param>
        /// <returns>A vector created as vector a minus vector b</returns>
        public static VectorN operator -(VectorN a, VectorN b)
        {
            if (a.n != b.n)
            {
                throw new ArgumentException("Vectors must be of same size");
            }

            VectorN r = new VectorN(a.n);
            for (int i = 0; i < a.n; i++)
            {
                r.x[i] = a.x[i] - b.x[i];
            }

            return r;
        }

        /// <summary>
        /// Multiplication of a vector by a real number from the left side
        /// </summary>
        /// <param name="c"> A real number</param>
        /// <param name="a"> A vector</param>
        /// <returns> A vector which components c * a[i]</returns>
        public static VectorN operator *(double c, VectorN a)
        {
            VectorN r = new VectorN(a.n);
            for (int i = 0; i < a.n; i++)
            {
                r.x[i] = c * a.x[i];
            }

            return r;
        }

        /// <summary>
        /// Multiplication of a vector by a real number from the right side
        /// </summary>
        /// <param name="a"> A vector</param>
        /// <param name="c"> A real number</param>
        /// <returns> A vector with compoonents a[i] * c</returns>
        public static VectorN operator *(VectorN a, double c)
        {
            return c * a;
        }

        /// <summary>
        /// Division operator. A real number is divided by a vector.
        /// </summary>
        /// <param name="a"> A vector. All components should be different from zero.</param>
        /// <param name="c"> A real number.</param>
        /// <returns></returns>
        public static VectorN operator /(double c, VectorN a)
        {
            for (int i = 0; i < a.n; i++)
            {
                if (a[i] == 0.0)
                {
                    throw new ArgumentOutOfRangeException("a", "All components must be non-zero");
                }
            }

            VectorN r = new VectorN(a.n);
            for (int i = 0; i < a.n; i++)
            {
                r.x[i] = c / a.x[i];
            }

            return r;
        }

        /// <summary>
        /// Division operator. A vector is divided by a real number.
        /// </summary>
        /// <param name="c"> A real number. Should be different from zero.</param>
        /// <param name="a"> A vector</param>
        /// <returns></returns>
        public static VectorN operator /(VectorN a, double c)
        {
            if (c == 0.0)
            {
                throw new ArgumentOutOfRangeException("c", "Must be non-zero");
            }

            double d = 1.0 / c;
            return d * a;
        }
    }
}
