﻿//-----------------------------------------------------------------------------
// InfoLinkReceiver.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Networking
{
    using System;
    using System.Net;
    using System.Net.Sockets;
    using System.Text;
    using System.Threading;

    /// <summary>
    /// Class representing an InfoLink UDP Receiver.
    /// </summary>
    public class InfoLinkReceiver : InfoLinkNetBase
    {
        /// <summary>
        /// The thread running state.
        /// </summary>
        private volatile bool running = false;

        /// <summary>
        /// The thread running the listener.
        /// </summary>
        Thread listenerThread;

        /// <summary>
        /// The handler delegate.
        /// </summary>
        ReceiverHandler handler;

        /// <summary>
        /// Delegate which handles the received InfoLink objects.
        /// </summary>
        /// <param name="infoLink"></param>
        public delegate void ReceiverHandler(InfoLink infoLink);

        /// <summary>
        /// Creates am InfoLinkReceiver to receive InfoLink objects 
        /// via UDP on the default port. 
        /// </summary>
        /// <param name="handler">The callback to handle received InfoLinks.</param>
        public InfoLinkReceiver(ReceiverHandler handler)
        {
            this.handler = handler;
            this.port = InfoLinkNetDefaultPort;
        }

        /// <summary>
        /// Creates am InfoLinkReceiver to receive InfoLink objects 
        /// via UDP on the specified port.
        /// </summary>
        /// <param name="handler">The callback to handle received InfoLinks.</param>
        /// <param name="port">The port to use.</param>
        public InfoLinkReceiver(ReceiverHandler handler, int port)
        {
            this.handler = handler;
            this.port = port;
        }

        /// <summary>
        /// Destructor for InfoLinkReceiver.
        /// </summary>
        ~InfoLinkReceiver()
        {
            if (this.Connected)
            {
                this.Close();
            }
        }

        /// <summary>
        /// Indicates wether the listener thread is running.
        /// </summary>
        public bool Running
        {
            get { return this.running; }
            set { this.running = value; }
        }

        /// <summary>
        /// Start the listener thread.
        /// </summary>
        public void StartListener()
        {
            if (this.Connected)
            {
                this.Close();
            }

            this.Open();
            this.listenerThread = new Thread(new ThreadStart(RunListenerThread));
            this.listenerThread.Name = "InfoLink Receiver";
            this.listenerThread.IsBackground = true;
            this.listenerThread.Start();
        }

        /// <summary>
        /// Stop the listener thread.
        /// </summary>
        public void StopListener()
        {
            if (this.Connected)
            {
                this.Close();
            }
        }

        /// <summary>
        /// Open the UDP client connection.
        /// </summary>
        private void Open()
        {
            this.udpClient = new UdpClient(this.port);
            this.endPoint = new IPEndPoint(IPAddress.Any, this.port);
            this.Connected = true;
        }

        /// <summary>
        /// Close UDP client connection and stop the listener thread.
        /// </summary>
        private void Close()
        {
            try
            {
                this.running = false;
                this.udpClient.Close();
                this.listenerThread = null;
            }
            finally
            {
                this.Connected = false;
            }
        }

        /// <summary>
        /// Receive datagrams in a loop and try to deserialize
        /// them into InfoLink objects.
        /// </summary>
        private void RunListenerThread()
        {
            this.running = this.Connected;

            while (this.running)
            {
                // Receive a datagram
                byte[] bytes = null;
                try
                {
                    bytes = this.udpClient.Receive(ref this.endPoint);
                }
                catch
                {
                    // Assume the socket was closed, 
                    // so we can exit the thread loop
                    this.running = false;
                }

                // Process datagram
                if (null != bytes && this.running)
                {
                    string message =
                        Encoding.UTF8.GetString(bytes, 0, bytes.Length);

                    InfoLink infoLink = null;
                    try
                    {
                        infoLink = InfoLinkSerializer.Deserialize(message);
                    }
                    finally
                    {
                        if (null != infoLink)
                        {
                            // Use data via delegate callback
                            this.handler(infoLink);
                        }
                    }
                }
            }
        }
    }
}
