﻿//-----------------------------------------------------------------------------
// PlayTrulyRandom.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Networking
{
    using System;
    using System.Net;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.IO;

    /// <summary>
    /// Interface to the playtrulyrandom.com webservices.
    /// </summary>
    public class PlayTrulyRandom
    {
        /// <summary>
        /// The base URL to submit entropy bits.
        /// </summary>
        private const string baseUrl =
            "http://www.playtrulyrandom.com/";

        /// <summary>
        /// The customized URL to use to submit entropy bits.
        /// </summary>
        private string submitUrl;

        /// <summary>
        /// The customized URL to use to submit entropy bits.
        /// </summary>
        private string retrieveUrl;

        /// <summary>
        /// The customized URL to use to submit usage info.
        /// </summary>
        private string usageUrl;

        /// <summary>
        /// The name of the agent for tracking and authentication
        /// with playtrulyrandom.com webservice.
        /// Is also used as agent name for the http request.
        /// </summary>
        private string agentName;

        /// <summary>
        /// The name of the user for tracking and authentication
        /// with playtrulyrandom.com webservice.
        /// </summary>
        private string userName;

        /// <summary>
        /// The size of the retrieval bit pool in bits.
        /// </summary>
        private int retrievePoolSize;

        /// <summary>
        /// The retrieve bit pool cache (string of bits).
        /// </summary>
        private string retrievePoolCache;

        /// <summary>
        /// The current position of the next bit in the pool.
        /// </summary>
        private int retrievePoolPos;

        /// <summary>
        /// The last message from an online call.
        /// </summary>
        private string lastMessage = string.Empty;

        /// <summary>
        /// Flag indicating that an error occured during the last online call. 
        /// </summary>
        private bool error = false;

        /// <summary>
        /// Creates an instance of the PlayTrulyRandom access object for an agent.
        /// A new username is created as GUID string.
        /// A retrieve pool of 256 bits is set.
        /// </summary>
        /// <param name="agentName">The agent name.</param>
        public PlayTrulyRandom(string agentName)
            : this(agentName, Guid.NewGuid().ToString(), 256)
        {
        }

        /// <summary>
        /// Creates an instance of the PlayTrulyRandom access object for an agent.
        /// A new username is created as GUID string.
        /// </summary>
        /// <param name="agentName">The agent name.</param>
        /// <param name="retrievePoolSize">
        /// The retrieve pool size. 
        /// Must be in the range [8,1024].
        /// </param>
        public PlayTrulyRandom(string agentName, int retrievePoolSize)
            : this(agentName, Guid.NewGuid().ToString(), retrievePoolSize)
        {
        }

        /// <summary>
        /// Creates an instance of the PlayTrulyRandom access object for an agent
        /// and the specific user. A retrieve pool of 256 bits is set.
        /// </summary>
        /// <param name="agentName">The agent name.</param>
        /// <param name="userName">The user name.</param>
        public PlayTrulyRandom(string agentName, string userName)
            : this(agentName, userName, 256)
        {
        }

        /// <summary>
        /// Creates an instance of the PlayTrulyRandom submittor for an agent.
        /// </summary>
        /// <param name="agentName">The agent name.</param>
        /// <param name="userName">The user name.</param>
        /// <param name="retrievePoolSize">
        /// The retrieve pool size. 
        /// Must be in the range [8,1024].
        /// </param>
        public PlayTrulyRandom(string agentName, string userName, int retrievePoolSize)
        {
            // Check arguments
            if (string.IsNullOrEmpty(agentName))
            {
                throw new ArgumentNullException(
                    "agentName",
                    "Parameter cannot be null or empty.");
            }

            if (null == userName)
            {
                throw new ArgumentNullException("userName");
            }

            if ((retrievePoolSize < 2) || (retrievePoolSize > 1024))
            {
                throw new ArgumentOutOfRangeException(
                    "retrievePoolSize",
                    "Must be in the range [8,1024]");
            }
            // Store parameters
            this.agentName = Uri.EscapeDataString(agentName);
            this.userName = Uri.EscapeDataString(userName);
            this.retrievePoolSize = retrievePoolSize;

            string commonPart = "?source=" + agentName + "&user=" + userName + "&";

            // Build submission URL without bits data value:
            // Example: http://www.playtrulyrandom.com/submit.php?source=pendulumgame_v1&user=xyz&bits=011001
            this.submitUrl = baseUrl + "submit.php" + commonPart + "bits=";

            // Build retrieval URL without n data value:
            // Example: http://www.playtrulyrandom.com/retrieve.php?source=montygame_v1&user=xyz&n=32
            this.retrieveUrl = baseUrl + "retrieve.php" + commonPart + "n=";

            // Build usage URL without duration data value:
            // Example: http://www.playtrulyrandom.com/retrieve.php?source=montygame_v1&user=xyz&duration=32
            this.usageUrl = baseUrl + "usage.php" + commonPart + "duration=";

            // Reset retrieve pool state
            this.retrievePoolCache = string.Empty;
            this.retrievePoolPos = -1;
        }

        /// <summary>
        /// Gets the last message from the last online call.
        /// </summary>
        public string LastMessage
        {
            get { return this.lastMessage; }
        }

        /// <summary>
        /// Gets the error flag from the last online call.
        /// </summary>
        public bool Error
        {
            get { return this.error; }
        }

        /// <summary>
        /// Submits the bits to the online collection service.
        /// Updates error flag.
        /// </summary>
        /// <param name="param">
        /// The string representation of a bit string 
        /// consisting of 0 and 1's to submit.
        /// </param>
        public void OnlineSubmit(object param)
        {
            // The bits to submit
            string bitsToSumit = (string)param;

            // Reset error message
            this.lastMessage = string.Empty;
            this.error = false;

            // Check if we have to submit anything
            if (!String.IsNullOrEmpty(bitsToSumit))
            {
                // Create REST url
                Uri url = new Uri(this.submitUrl + Uri.EscapeDataString(bitsToSumit));

                // Submit query, ignore result
                string result = UrlGet(url);
            }
        }

        /// <summary>
        /// Submits the usage data to the online collection service.
        /// Updates error flag.
        /// </summary>
        /// <param name="param">
        /// The integer representation number of seconds of usage to submit.
        /// </param>
        public void OnlineUsage(object param)
        {
            // The seconds to submit
            int usageSeconds = (int)param;

            // Reset error message
            this.lastMessage = string.Empty;
            this.error = false;

            // Check if we have to submit anything
            if (usageSeconds > 0)
            {
                // Create REST url
                Uri url = new Uri(this.usageUrl + usageSeconds.ToString());

                // Submit query, ignore result
                string result = UrlGet(url);
            }
        }

        /// <summary>
        /// Register with the webservice. Syncronous call.
        /// May update error flag.
        /// </summary>
        public void RegisterAgent()
        {
            // Reset error message
            this.lastMessage = string.Empty;
            this.error = false;

            // Create REST url
            Uri url = new Uri(this.usageUrl + "1&hrngmode=1");

            // Submit query, ignore result
            string result = UrlGet(url);
        }

        /// <summary>
        /// Retrieve bits from the online collection service.
        /// Updates error flag.
        /// </summary>
        /// <param name="numberOfBits">
        /// The number of bits to read.
        /// </param>
        /// <returns>A bit string consisting of 0 and 1 characters.</returns>
        public string OnlineRetrieve(int numberOfBits)
        {
            // Check input parameters
            if ((numberOfBits < 1) || (numberOfBits > 1024))
            {
                throw new ArgumentOutOfRangeException(
                    "numberBits",
                    "Requested number must be between 1 and 1024");
            }

            // Reset error message
            this.lastMessage = string.Empty;
            this.error = false;

            // Create REST url
            Uri url = new Uri(this.retrieveUrl + Uri.EscapeDataString(numberOfBits.ToString()));

            // Submit query
            string result = UrlGet(url);

            // Result contains bits
            return result;
        }

        /// <summary>
        /// Retrieve bits from internal bit pool filled by the
        /// online collection service. May make one or more calls
        /// to the online webservice.
        /// </summary>
        /// <param name="numberOfBits">
        /// The number of bits to read.
        /// </param>
        /// <returns>A bit string consisting of 0 and 1 characters.</returns>
        public string PoolRetrieve(int numberOfBits)
        {
            // Reset result string
            string retrievedBits = string.Empty;

            // Loop to copy bits
            int bitsToCopy = numberOfBits;
            while (bitsToCopy > 0)
            {
                // Check if we need to fill the cache
                if ((this.retrievePoolPos < 0) || (this.retrievePoolPos == this.retrievePoolSize))
                {
                    // Fill the cache from online source
                    this.retrievePoolCache = this.OnlineRetrieve(this.retrievePoolSize);
                    this.retrievePoolPos = 0;

                    if (this.Error)
                    {
                        throw new Exception(
                            "Could not retrieve random data. " +
                            this.LastMessage);
                    }

                    if (this.retrievePoolCache.Length != this.retrievePoolSize)
                    {
                        throw new Exception(
                            "Could not retrieve the required number of bits. " +
                            "Expecting: " + this.retrievePoolSize + " bits, " +
                            "Got: " + this.retrievePoolCache.Length + " bits.");
                    }
                }

                // Now copy cached pool to output string
                int bitsLeftInPool = this.retrievePoolSize - this.retrievePoolPos;
                if (bitsLeftInPool >= bitsToCopy)
                {
                    // Have enough bits in cache, use them and update pos
                    retrievedBits += this.retrievePoolCache.Substring(this.retrievePoolPos, bitsToCopy);
                    this.retrievePoolPos += bitsToCopy;
                    bitsToCopy = 0;
                }
                else
                {
                    // Don't have enough bits in cache, use what we have
                    retrievedBits += this.retrievePoolCache.Substring(this.retrievePoolPos, bitsLeftInPool);
                    bitsToCopy -= bitsLeftInPool;

                    // Mark pos to load more
                    this.retrievePoolPos = -1;
                }
            }

            return retrievedBits;
        }

        /// <summary>
        /// Generate random number within a given range [min,max] by 
        /// retrieving bits from the webservice and mapping them
        /// into the range. This may require multiple 
        /// pool or webservice requests.
        /// </summary>
        /// <param name="minValue">
        /// The inclusive minimum value to be generated.
        /// </param>
        /// <param name="maxValue">
        /// The inclusive maximum value to be generated. 
        /// maxValue must be greater or equals than minValue.
        /// </param>
        /// <param name="retrievedBits">Number of bits retrieved
        /// to generate random number.</param>
        /// <returns>A random number within the range.</returns>
        public int Next(int minValue, int maxValue, out int retrievedBits)
        {
            if (maxValue < minValue)
            {
                throw new ArgumentException(
                    "maxValue must be greater or equals than minValue",
                    "maxValue");
            }

            int result;
            retrievedBits = 0;

            // Assume min value
            result = minValue;

            // Determine difference plus one
            long delta = (long)maxValue - (long)minValue;
            if (delta < 1)
            {
                return result;
            }

            // Determine number of bits needed
            int numberOfBits = 1;
            long currentValue = 2;
            long deltaPlus = delta + 1;
            while (deltaPlus > currentValue)
            {
                numberOfBits++;
                currentValue *= 2;
            }

            // We MUST retrieve new bits until we have a number 
            // that is within the requested delta range
            // to ensure uniformity; a modulus calculation is
            // NOT appropriate here.
            long randomNumber = 0;
            do
            {
                // Get a bit string
                string randomBits = this.PoolRetrieve(numberOfBits);
                retrievedBits += numberOfBits;
                if (randomBits.Length != numberOfBits)
                {
                    this.lastMessage = String.Format(
                        "Could not retrieve the required number of bits. " +
                        "Expecting: {0} bits, " +
                        "Got: {1} bits.",
                        numberOfBits,
                        randomBits.Length);
                    this.error = true;
                    throw new ApplicationException(this.lastMessage);
                }

                // Convert to number
                randomNumber = Convert.ToInt64(randomBits, 2);
            } while (randomNumber > delta);

            // Calculate result
            result = (int)((long)minValue + randomNumber);

            return result;
        }

        /// <summary>
        /// Submit a REST url to webservice.
        /// Does not throw an exception, but sets a flag and message if
        /// an error occurs.
        /// </summary>
        /// <param name="url">The URL to submit.</param>
        /// <returns>The data read form the request.</returns>
        private string UrlGet(Uri url)
        {
            string retrievedText = string.Empty;
            HttpWebRequest request;
            HttpWebResponse response = null;
            try
            {
                // Create the web request
                request = WebRequest.Create(url) as HttpWebRequest;

                // Set request parameters
                request.UserAgent = this.agentName;
                request.KeepAlive = false;

                // Set timeout to 15 seconds
                request.Timeout = 15 * 1000;

                // Get response
                response = request.GetResponse() as HttpWebResponse;
                if (request.HaveResponse == true && response != null)
                {
                    // Get the response stream
                    StreamReader reader = new StreamReader(response.GetResponseStream());

                    // Capture the output
                    retrievedText = reader.ReadToEnd();
                }
            }
            catch (WebException wex)
            {
                // This exception will be raised if the server didn't return 200 - OK
                // Try to retrieve more information about the network error
                if (wex.Response != null)
                {
                    using (HttpWebResponse errorResponse = (HttpWebResponse)wex.Response)
                    {
                        this.lastMessage = String.Format("Network error '{0}' {1} ({2:d}).",
                            errorResponse.StatusDescription,
                            errorResponse.StatusCode,
                            errorResponse.StatusCode);
                    }

                    this.error = true;
                }
            }
            catch (Exception e)
            {
                this.lastMessage = e.Message;

                this.error = true;
            }
            finally
            {
                if (response != null)
                {
                    response.Close();
                }
            }

            return retrievedText;
        }
    }
}
