//-----------------------------------------------------------------------------
// CircularObjectSimulation.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.PhysicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using Microsoft.Xna.Framework;

    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;

    /// <summary>
    /// Represents a moving circular object in 2D space under gravity influence.
    /// </summary>
    public class CircularObjectSimulation
    {
        /// <summary>
        /// The acceleration function for the circular obejct.
        /// </summary>
        private CircularObjectAcceleration circularObjectAcceleration;
        
        /// <summary>
        /// The integrator for the circular object.
        /// </summary>
        private NystromIntegrator circularObjectIntegrator;

        /// <summary>
        /// Preallocated return variable
        /// </summary>
        private Vector2 circularObjectPosition;

        /// <summary>
        /// Animatation parameter (time).
        /// </summary>
        private double t;

        /// <summary>
        /// Stepsize for integrator (time).
        /// </summary>
        private double h;

        /// <summary>
        ///  Physical state: 2D position.
        /// </summary>
        private VectorN p;

        /// <summary>
        /// Physical state: 2D speed.
        /// </summary>
        private VectorN v;

        /// <summary>
        /// Physical state: 2D acceleration.
        /// </summary>
        private VectorN acceleration;

        /// <summary>
        /// The mass of the object.
        /// </summary>
        private double mass;

        /// <summary>
        /// Tracks the requirement for the integrator to be reset after
        /// a change of initial conditions.
        /// </summary>
        private bool integratorNeedsReset;

        /// <summary>
        /// Constructor for animated circular object.
        /// </summary>
        /// <param name="objectDiameter">Object (ball) diameter</param>
        /// <param name="objectDensity">Object (ball) material density</param>
        /// <param name="gravity">Acceleration due to gravity</param>
        /// <param name="atmosphereDensity">The density of the atmosphere</param>
        public CircularObjectSimulation(double objectDiameter, double objectDensity, double gravity, double atmosphereDensity)
        {
            // Time and stepsize
            t = 0;
            h = 0.01;

            // Drag coefficient (cD)
            double dragSphere = DragCoefficient.Sphere;

            // Ball mass = density * volume
            this.mass = objectDensity * Volume.Sphere(objectDiameter / 2.0);

            // Ball acceleration
            this.circularObjectAcceleration =
                new CircularObjectAcceleration(objectDiameter, objectDensity, dragSphere, -gravity, atmosphereDensity);

            // Reset state
            this.p = new VectorN(2);
            this.v = new VectorN(2);

            // Create integrator
            this.circularObjectIntegrator = new NystromIntegrator(
                (ISecondDerivative)this.circularObjectAcceleration, 
                this.h, 
                this.p, 
                this.v);
            this.integratorNeedsReset = false;

            // Allocate physical state variables as cache
            circularObjectPosition = new Vector2();
        }

        /// <summary>
        /// Resets the initial conditions (position and velocity) for the
        /// object to be at rest.
        /// </summary>
        public void InitialConditionsAtRest()
        {
            SetInitialConditions(0.0, 0.0, 0.0, 0.0);
        }

        /// <summary>
        /// Sets the initial conditions (position and speed) for the 
        /// the object simulation.
        /// </summary>
        /// <param name="x">The x coordinate of the object.</param>
        /// <param name="y">The y coordinate of the object.</param>
        /// <param name="vx">The x component of the objects speed.</param>
        /// <param name="vy">The y component of hte objects speed.</param>
        public void SetInitialConditions(double x, double y, double vx, double vy)
        {
            // Store values
            this.p[0] = x;
            this.p[1] = y;
            this.v[0] = vx;
            this.v[1] = vy;

            // Mark for integrator to require a reset
            this.integratorNeedsReset = true;
        }

        /// <summary>
        /// Animate the circular object.
        /// </summary>
        public void Animate()
        {
            // Check if integrator needs to be reset
            if (this.integratorNeedsReset)
            {
                this.acceleration = circularObjectIntegrator.Reset(this.p, this.v);
            }

            // Animate ball
            // double t; VectorN p, v, acceleration;
            circularObjectIntegrator.Step(out t, out p, out v, out acceleration);
        }

        /// <summary>
        /// Drive ball by force transfer, changing its speed.
        /// </summary>
        /// <param name="force">The force to apply to the object.</param>
        public void Drive(Vector2 force)
        {
            v[0] = this.h * force.X * this.mass;
            v[1] = this.h * force.Y * this.mass;
        }

        /// <summary>
        /// Set the ball positions to a point.
        /// Resets speed to zero.
        /// </summary>
        /// <param name="position">Position where to place the ball.</param>
        public void SetPosition(Vector2 position)
        {
            // Set position
            p[0] = position.X;
            p[1] = position.Y;

            // Reset speed
            v[0] = 0.0f;
            v[1] = 0.0f;
        }

        /// <summary>
        /// Set the ball positions to a point which is at a particular scale.
        /// </summary>
        /// <param name="position">Scaled position where to place the ball.</param>
        /// <param name="scale">The scale of the ball (pixels/meter).</param>
        public void SetPosition(Vector2 position, double scale)
        {
            // Set position
            if (scale != 0.0)
            {
                p[0] = position.X / scale;
                p[1] = position.Y / scale;
            }

            // Reset speed
            v[0] = 0.0f;
            v[1] = 0.0f;
        }

        /// <summary>
        /// Get object positions.
        /// </summary>
        /// <returns>The position.</returns>
        public Vector2 GetPosition()
        {
            // Set position
            circularObjectPosition.X = (float)(p[0]);
            circularObjectPosition.Y = (float)(p[1]);

            return circularObjectPosition;
        }

        /// <summary>
        /// Get object positions relative to a particular scale.
        /// </summary>
        /// <param name="scale">The scale of the ball (pixels/meter).</param>
        /// <returns>The scaled position.</returns>
        public Vector2 GetPosition(double scale)
        {
            // Set position
            circularObjectPosition.X = (float)(scale * p[0]);
            circularObjectPosition.Y = (float)(scale * p[1]);

            return circularObjectPosition;
        }
    }
}

