//-----------------------------------------------------------------------------
// DoubleRegularPendulumSimulation.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.PhysicalElements
{
    using System;
    using System.Collections.Generic;
    using System.Text;
    
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    using NewGamePhysics.Utilities;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Physics;

    /// <summary>
    /// Simulation of a regular double pendulum.
    /// </summary>
    public class DoubleRegularPendulumSimulation : DoublePendulumSimulationBase
    {
        /// <summary>
        /// Constructor for animated double pendulum using a standard stepsize.
        /// </summary>
        /// <param name="origin">Position or anchor of pendulum.</param>
        /// <param name="l1">Length of the first rod.</param>
        /// <param name="m1">Mass of the first pendulum component.</param>
        /// <param name="l2">Length of the second rod.</param>
        /// <param name="m2">Mass of the second pendulum component</param>
        /// <param name="g">The gravitational acceleration to apply.</param>
        /// <param name="f">The frictional model to use for the hinges.</param>
        public DoubleRegularPendulumSimulation(
            Vector2 origin,
            double l1, 
            double m1, 
            double l2, 
            double m2, 
            double g, 
            RotationalFrictionType f)
        {
            // Origin
            this.origin = origin;

            // Time and stepsize
            this.T = 0.0;
            this.H = 0.01;

            // Construct pendulum
            this.PendulumAcceleration =
                (ISecondDerivative)new DoubleRegularPendulumAcceleration(l1, m1, l2, m2, g);
            
            // Reset initial conditions
            this.theta = new VectorN(2);
            this.omega = new VectorN(2);
            this.SetInitialConditionsAtRest();

            // Initialize friction model
            this.FrictionModel = new RotationalFrictionModel(f);
        }

        /// <summary>
        /// Resets the initial conditions (angles and rotational speeds) for 
        /// the pendulum simulation to a be at rest.
        /// </summary>
        public override void SetInitialConditionsAtRest()
        {
            SetInitialConditions(0.0, 0.0, 0.0, 0.0);
        }

        /// <summary>
        /// Calculate pedulum positions relative.
        /// </summary>
        /// <returns>The pendulum positions.</returns>
        public override Vector2[] GetPosition()
        {
            // Allocate cache for position state of the pendulum components
            Vector2[] pendulumPoints = new Vector2[3];
            for (int i = 0; i < pendulumPoints.Length; i++)
            {
                pendulumPoints[i] = new Vector2();
            }

            // Set angle for top component
            double angle = this.theta[0];

            // Set anchor/top of pendulum
            pendulumPoints[0].X = this.origin.X;
            pendulumPoints[0].Y = this.origin.Y;

            double[] pendulumSize = this.GetSize();

            // Calculate top component positions
            pendulumPoints[1].X = 
                (float)(pendulumPoints[0].X + (pendulumSize[0] * Math.Sin(angle)));
            pendulumPoints[1].Y = 
                (float)(pendulumPoints[0].Y + (pendulumSize[0] * Math.Cos(angle)));

            // Set angle for bottom component
            angle = this.theta[1];

            // Calculate bottom component positions
            pendulumPoints[2].X = 
                (float)(pendulumPoints[1].X + (pendulumSize[1] * Math.Sin(angle)));
            pendulumPoints[2].Y = 
                (float)(pendulumPoints[1].Y + (pendulumSize[1] * Math.Cos(angle)));

            return pendulumPoints;
        }

        /// <summary>
        /// Calculate pedulum positions relative to a screen origin and scale.
        /// </summary>
        /// <param name="screenOrigin">Position where to anchor the pendulum.</param>
        /// <param name="screenScale">The scale of the pendulum (pixels/meter).</param>
        /// /// <returns>The pendulum positions.</returns>
        public override Vector2[] GetPosition(Vector2 screenOrigin, double screenScale)
        {
            // Update
            Vector2[] pendulumPoints = this.GetPosition();

            // Scale in place
            for (int i=0; i<pendulumPoints.Length; i++)
            {
                pendulumPoints[i].X = screenOrigin.X + (float)(screenScale * pendulumPoints[i].X);
                pendulumPoints[i].Y = screenOrigin.Y + (float)(screenScale * pendulumPoints[i].Y);
            }

            return pendulumPoints;
        }

        /// <summary>
        /// Calculate the 2 sizes of each bar .
        /// </summary>
        /// <returns>Array containing two doubles.
        /// First value is for the top bar, second for the bottom bar.</returns>
        public override double[] GetSize()
        {
            DoubleRegularPendulumAcceleration pendulumAcceleration =
                this.PendulumAcceleration as DoubleRegularPendulumAcceleration;

            double[] pendulumSize = new double[2];
            pendulumSize[0] = pendulumAcceleration.L1;
            pendulumSize[1] = pendulumAcceleration.L2;

            return pendulumSize;
        }

        /// <summary>
        /// Calculate the 2 sizes of each bar with the scale applied.
        /// </summary>
        /// <param name="scale">The scale factor for the size calculation.</param>
        /// <returns>Array containing two doubles.
        /// First value is for the top bar, second for the bottom bar.</returns>
        public override double[] GetSize(double scale)
        {
            double[] size = this.GetSize();
            for (int i = 0; i < size.Length; i++)
            {
                size[i] *= scale;
            }

            return size;
        }

        /// <summary>
        /// Calculates the current total energy of the system.
        /// </summary>
        /// <returns>The energy value.</returns>
        public override double GetEnergy()
        {
            DoubleRegularPendulumAcceleration pendulumAcceleration =
                this.PendulumAcceleration as DoubleRegularPendulumAcceleration;

            double f1 = (1 - Math.Cos(theta[0]));
            double f2 = (1 - Math.Cos(theta[1]));
            double v = pendulumAcceleration.M1 * pendulumAcceleration.G *  (pendulumAcceleration.L1 * f1) +
                       pendulumAcceleration.M2 * pendulumAcceleration.G * ((pendulumAcceleration.L1 * f1) + (pendulumAcceleration.L2 * f2));
            double k = 0.5 * (pendulumAcceleration.M1 + pendulumAcceleration.M2) * pendulumAcceleration.L1 * pendulumAcceleration.L1 * omega[0] * omega[0] +
                       0.5 * pendulumAcceleration.M2 * pendulumAcceleration.L2 * pendulumAcceleration.L2 * omega[1] * omega[1] +
                       pendulumAcceleration.M2 * pendulumAcceleration.L1 * pendulumAcceleration.L2 * Math.Cos(theta[0] - theta[1]) * omega[0] * omega[1];
            return (v + k);
        }
    }
}
