// --------------------------------------------------------
// CircularObjectAcceleration.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using NewGamePhysics.Mathematics;

   /// <summary>
    /// An object defining the acceleration of an object of circular shape (i.e. ball)
    /// in a gas (i.e. air) under the influence of gravity.
    /// </summary>
    class CircularObjectAcceleration : ISecondDerivative
    {
        double dB;      // Object diameter
        double roB;     // Object material density
        double cD;      // Drag coefficient
        double g;       // Acceleration due to gravity
        double roA;     // Air density
        double mB;      // Ball mass
        double aB;      // Area of the ball cross-section

        /// <summary>
        /// A constructor of circular object
        /// </summary>
        /// <param name="dB">Ball diameter</param>
        /// <param name="roB">Ball material density</param>
        /// <param name="cD">Drag coefficient</param>
        /// <param name="g">Acceleration due to gravity</param>
        /// <param name="roA">Air density</param>
        public CircularObjectAcceleration(double dB, double roB, double cD, double g, double roA)
        {
            this.dB = dB;
            this.roB = roB;
            this.cD = cD;
            this.g = g;
            this.roA = roA;

            double v = Math.PI * Math.Pow(dB, 3) / 6;
            mB = roB * v;
            aB = 0.25 * Math.PI * dB * dB;
        }

        /// <summary>
        /// A method to calculate the vector of acceleration of a
        /// circular object in air.
        /// </summary>
        /// <param name="t"> Time</param>
        /// <param name="p"> VectorN of positions</param>
        /// <param name="v"> VectorN of velocities</param>
        /// <returns> VectorN of accelerations</returns>
        VectorN ISecondDerivative.GetValue(double t, VectorN p, VectorN v)
        {
            double vx = v[0];
            double vy = v[1];
            double vxSqr = vx * vx;
            double vySqr = vy * vy;
            double vSqr = vxSqr + vySqr;
            double vTotal = Math.Sqrt(vSqr);

            double cosAlpha = 1;
            double cosBeta = 0;
            if (vTotal > 1.0e-12)
            {
                cosAlpha = vx / vTotal;
                cosBeta = vy / vTotal;
            }

            double drag = 0.5 * cD * roA * vSqr * aB;
            double dragX = -drag * cosAlpha;
            double dragY = -drag * cosBeta;

            VectorN force = new VectorN(2);
            force[0] = dragX;
            force[1] = -mB * g + dragY;

            VectorN acceleration = force/mB;
            return acceleration;
        }
    }
}
