// --------------------------------------------------------
// DoubleRegularPendulumAcceleration.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using NewGamePhysics.Mathematics;

    /// <summary>
    /// An object defining the acceleration of a 2D double pendulum consisting of
    /// two masses connected by a massless rod under influence of gravity.
    /// Reference: myphysicslab.com/dbl_pendulum.html
    /// </summary>
    public class DoubleRegularPendulumAcceleration : ISecondDerivative
    {
        private double l1;      // length of rod of top pendulum (constant)
        private double m1;      // mass of top pendulum (constant)
        private double l2;      // length of rod of bottom pendulum (constant)
        private double m2;      // mass of bottom pendulum (constant)
        private double g;       // gravitational constant
       
        /// <summary>
        /// A constructor of the DoubleRegularPendulum object
        /// </summary>
        public DoubleRegularPendulumAcceleration(double l1, double m1, double l2, double m2, double g)
        {
            this.l1 = l1;
            this.m1 = m1;
            this.l2 = l2;
            this.m2 = m2;
            this.g = g;
        }

        /// <summary>
        /// Returns size of top rod.
        /// </summary>
        public double L1
        {
            get
            {
                return this.l1;
            }
        }

        /// <summary>
        /// Returns weight of the top pendulum.
        /// </summary>
        public double M1
        {
            get
            {
                return this.m1;
            }
        }

        /// <summary>
        /// Returns size of bottom rod.
        /// </summary>
        public double L2
        {
            get
            {
                return this.l2;
            }
        }

        /// <summary>
        /// Returns weight of the bottom pendulum.
        /// </summary>
        public double M2
        {
            get
            {
                return this.m2;
            }
        }

        /// <summary>
        /// Sets the gravitational constant.
        /// </summary>
        public double G
        {
            get
            {
                return this.g;
            }

            set
            {
                this.g = value;
            }
        }

        /// <summary>
        /// A method to calculate the vector of accelerations.
        /// </summary>
        /// <param name="t"> Time</param>
        /// <param name="theta"> VectorN of angle of pendulum (0=vertically downwards, counter-clockwise is positive)</param>
        /// <param name="omega"> VectorN of angular velocities of rod</param>
        /// <returns> VectorN of accelerations</returns>
        VectorN ISecondDerivative.GetValue(double t, VectorN theta, VectorN omega)
        {
            double theta1 = theta[0];
            double theta2 = theta[1];
            double omega1 = omega[0];
            double omega2 = omega[1];

            VectorN acceleration = new VectorN(2);
            double theta_diff = theta1 - theta2;
            double mass_sum = m1 + m2;
            double mass_sum2 = mass_sum + m1;
            double denom_part = mass_sum2 - m2 * Math.Cos(2 * theta_diff);
            double sin_theta_diff = Math.Sin(theta_diff);
            double cos_theta_diff = Math.Cos(theta_diff);
            acceleration[0] = (-g * mass_sum2 * Math.Sin(theta1) - m2 * g * Math.Sin(theta_diff - theta2) - 2 * sin_theta_diff * m2 * (omega2 * omega2 * l2 + omega1 * omega1 * l1 * cos_theta_diff)) / (l1 * denom_part);
            acceleration[1] = ( 2 * sin_theta_diff * (omega1 * omega1 * l1 * mass_sum + g * mass_sum * Math.Cos(theta1) + omega2 * omega2 * l2 * m2 * cos_theta_diff) ) / ( l2 * denom_part );

            return acceleration;
        }        
    }
}
