﻿// --------------------------------------------------------
// GravityFieldCoefficient.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;

    /// <summary>
    /// Gravity Field Coefficient (GFC) used to represent a gravity models.
    /// Reference:
    /// http://icgem.gfz-potsdam.de/ICGEM/ICGEM.html
    /// </summary>
    public class GravityFieldCoefficient
    {
        /// <summary>
        /// L-order.
        /// </summary>
        private int l;

        /// <summary>
        /// M-order.
        /// </summary>
        private int m;

        /// <summary>
        /// C_lm coefficient.
        /// </summary>
        private double c;

        /// <summary>
        /// S_lm coefficient.
        /// </summary>
        private double s;

        /// <summary>
        /// Gets or set the L-order.
        /// </summary>
        public int L
        {
            get { return this.l; }
            set { this.l = value; }
        }

        /// <summary>
        /// Gets or sets the M-order.
        /// </summary>
        public int M
        {
            get { return this.m; }
            set { this.m = value; }
        }

        /// <summary>
        /// Gets or sets the C_lm coefficient.
        /// </summary>
        public double C
        {
            get { return this.c; }
            set { this.c = value; }
        }

        /// <summary>
        /// Gets or sets the S_lm coefficient.
        /// </summary>
        public double S
        {
            get { return this.s; }
            set { this.s = value; }
        }

        /// <summary>
        /// Creates new gcf coefficient.
        /// </summary>
        /// <param name="l">L-order</param>
        /// <param name="m">M-order</param>
        /// <param name="c">C_lm coeff</param>
        /// <param name="s">S_lm coeff</param>
        public GravityFieldCoefficient(int l, int m, double c, double s)
        {
            this.l = l;
            this.m = m;
            this.c = c;
            this.s = s;
        }

        /// <summary>
        /// String representation of gfc coefficient.
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            return String.Format(
                "gfc {0} {1} {2} {3}",
                this.l,
                this.m,
                this.c,
                this.s);
        }
    }
}
