﻿// --------------------------------------------------------
// GravityMarsMroModel.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Collections.Generic;
    using System.IO;

    using NewGamePhysics.Mathematics;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Model of earth's gravity g based on Mars Reonnaissance Orbiter (MRO)
    /// measurements and a higher-order gravity anomaly expansion model
    /// represented by a gridded map. Loader only supports GGMRO files.
    /// </summary>
    public class GravityMarsMroModel
    {
        /// <summary>
        /// Minimum latitude of anomaly map.
        /// </summary>
        private const int LatMin = -90;

        /// <summary>
        /// Maximum latitude of anomaly map.
        /// </summary>
        private const int LatMax = 90;

        /// <summary>
        /// Minimum longitude of anomaly map.
        /// </summary>
        private const int LonMin = 0;

        /// <summary>
        /// Maximum longitude of anomaly map.
        /// </summary>
        private const int LonMax = 360;

        /// <summary>
        /// Unit of acceleration, name: Galileo, 1cm/sec^2.
        /// </summary>
        public const double Gal = 0.01;

        /// <summary>
        /// Gravitational Constant (m^3 kb^-1 s^-2)
        /// </summary>
        public const double G = 6.67428E-11;

        /// <summary>
        /// Mars Gravity Constant (G * M) in km^3/s^-2
        /// Reference: ggmro model parameter file.
        /// </summary>
        public const double GM = 42828.35796;

        /// <summary>
        /// Mars Equatorial Radius in km.
        /// Reference: ggmro model parameter file.
        /// </summary>
        public const double R = 3397.0;

        /// <summary>
        /// Mars's rotational speed in rad/s^-1.
        /// Reference: ggmro model parameter file.
        /// </summary>
        public const double omega = 7.088218081e-5;

        /// <summary>
        /// Gravity anomaly value map at lat [-90,90] and long [0,360] 
        /// in one degree steps.
        /// </summary>
        internal double[,] map;

        /// <summary>
        /// Constructs an uninitialized gravity mars object.
        /// </summary>
        public GravityMarsMroModel()
        {
        }

        /// <summary>
        /// Constructs a gravity mars object and initializes it by
        /// loading a anomaly map.
        /// </summary>
        /// <param name="modelFilename">The anomaly map file to load.</param>
        public GravityMarsMroModel(string mapFilename)
        {
            this.LoadAnomalyMap(mapFilename);
        }

        /// <summary>
        /// Calculate the acceleration g from the normal gravity and anomaly
        /// map of Mars at the surface. (height is ignored)
        /// Requires a mro map file to be loaded.
        /// </summary>
        /// <param name="lat">Latitude (deg) in the range [-90,90]</param>
        /// <param name="lon">Longitude (deg) in the range [0,360]</param>
        /// <param name="h">Height (m) above sealevel in the range [0,100000].</param>
        /// <returns>Acceleration g.</returns>
        public double Calculate(double lat, double lon, double h)
        {
            if (null == this.map)
            {
                throw new ApplicationException(
                    "Anomaly map not initialized. Load one first.");
            }

            if ((lat < -90.0) || (lat > 90.0))
            {
                throw new ArgumentOutOfRangeException("lat");
            }

            if ((lon < 0.0) || (lon > 360.0))
            {
                throw new ArgumentOutOfRangeException("lon");
            }

            if ((h < 0.0) || (h > 100000.0))
            {
                throw new ArgumentOutOfRangeException("h");
            }

            // Calculate index position
            int intLat = (int)Math.Truncate(lat);
            int intLon = (int)Math.Truncate(lon);

            // Create bicubic interpolator matrix
            double[,] G = new double[4, 4];
            for (int i = -1; i <= 2; i++)
            {
                for (int j = -1; j <= 2; j++)
                {
                    int indexLat = (LatWrap(intLat + i) - LatMin) % 180;
                    int indexLon = (LonWrap(intLon + j) - LonMin) % 360;
                    G[i + 1, j + 1] = map[indexLat, indexLon];
                }
            }

            // Interpolate over grid
            Bicubic bicubic = new Bicubic(G);
            double x = Math.Abs(lat - (double)intLat);
            double y = lon - (double)intLon;
            double g = bicubic.Calc(x, y);

            // Scale from mgal units
            // One milligal equals 0.01 mm/s/s.
            g *= 1e-5;

            // Add normal gravity
            double g0 = GravityMarsNormalModel.Calculate(lat);
            g += g0;

            return g;
        }

        /// <summary>
        /// Load the GGMRO anomaly map file.
        /// </summary>
        /// <param name="filename">The mro anomaly map file to load.</param>
        public void LoadAnomalyMap(string mapFilename)
        {
            // Load binary data
            FileStream fileStream = new FileStream(mapFilename, FileMode.Open, FileAccess.Read);
            BinaryReader binaryReader = new BinaryReader(fileStream);
            long totalBytes = new FileInfo(mapFilename).Length;
            byte[] byteArray = binaryReader.ReadBytes((Int32)totalBytes);
            binaryReader.Close();
            fileStream.Close(); 
            
            // GGMRO Image format
            // LINES                      = 180
            // LINE_SAMPLES               = 360
            // SAMPLE_TYPE                = IEEE_REAL
            // SAMPLE_BITS                = 64
            this.map = new double[180, 360];
            int startIndex = 0;
            byte[] tempArray = new byte[8];
            for (int lat = 0; lat < 180; lat++)
            {
                for (int lon = 0; lon < 360; lon++)
                {
                    // Get sample
                    // See: http://pds.jpl.nasa.gov/documents/sr/Chapter03.pdf
                    // and http://pds.jpl.nasa.gov/documents/sr/stdref2003/AppendixC.pdf
                    double sample;
                    if (BitConverter.IsLittleEndian)
                    {
                        for (int i = 0; i < 8; i++)
                        {
                            tempArray[7 - i] = byteArray[startIndex + i];
                        }

                        sample = BitConverter.ToDouble(tempArray, 0);
                    }
                    else
                    {
                        sample = BitConverter.ToDouble(byteArray, startIndex);
                    }

                    // Data format
                    // OFFSET                     = 0.0E+00
                    // SCALING_FACTOR             = 1.0E+00
                    // The values can be obtained by multiplying the sample in
                    // the map by SCALING_FACTOR and then adding OFFSET.  
                    double value = (1.0 * sample + 0.0);
                    this.map[lat, lon] = value ;

                    // Update source index
                    startIndex += (64 / 8);
                }
            }
        }

        /// <summary>
        /// Dispose of the current anomaly map.
        /// </summary>
        public void UnloadAnomalyMap()
        {
            this.map = null;
        }

        /// <summary>
        /// Wrap latitude value into range [-90,90].
        /// </summary>
        /// <param name="lat">Integer latitude value.</param>
        /// <returns>Wrapped latitude value.</returns>
        private int LatWrap(int lat)
        {
            while (lat < LatMin) 
            {
                lat =+ (LatMax - LatMin);
            } 

            while (lat > LatMax)
            {
                lat -= (LatMax - LatMin);
            }
 
            return lat;
        }

        /// <summary>
        /// Wrap longitude value into range [0,360].
        /// </summary>
        /// <param name="lat">Integer longitude value.</param>
        /// <returns>Wrapped longitude value.</returns>
        private int LonWrap(int lon)
        {
            while (lon < LonMin)
            {
                lon += (LonMax - LonMin);
            }

            while (lon > LonMax)
            {
                lon -= (LonMax - LonMin);
            }

            return lon;
        }
    }
}

