﻿// --------------------------------------------------------
// GravityMarsNormalModel.cs
// (A. Schiffler, 2009)
// --------------------------------------------------------

namespace NewGamePhysics.Physics
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// Model of earth's gravity g based on an elliptical shape.
    /// </summary>
    public class GravityMarsNormalModel
    {

        // Mars Equatorial gravity (m/sec2)
        // Reference: 
        // http://www.wolframalpha.com/input/?i=equatorial+gravity+on+mars
        public const double EquatorialGravity = 3.71;

        // Mars Ellipsoid Semi-Major Axis (km)
        // Reference: 
        // http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        public const double SemiMajorAxis = 3396.0;

        // Mars Ellipsoid Semi-Major Axis (JPL) (km)
        // Reference: 
        // http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        public const double SemiMajorAxisJpl = 3397.0;

        // Mars Ellipsoid Gravity constant (m**3/s**2)
        // Reference: 
        // http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        public const double GM = 4.282837519531400e+13;

        // Mars Ellipsoid Gravity constant (JPL) (m**3/s**2)
        // Reference: 
        // http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        public const double GMJpl = 42828.35796; // km**3/s**2

        // Mars Ellipsoid Rotation rate (rad/s)
        // Reference: 
        // http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        public const double RotationRate = 7.088218066303858e-05; // rad/s

        // Mars Ellipsoid Rotation rate (JPL) (rad/s)
        // Reference: 
        // http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        public const double RotationRateJpl = 7.088218081e-5; // rad/s

        // Mars Ellipsoid Inverse of Flattening
        // Reference: 
        // http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        public const double InverseFlattening = 196.877360;

        // Mars Ellipsoid Flattening
        // Reference: 
        // http://pds-geosciences.wustl.edu/mro/mro-m-rss-5-sdp-v1/mrors_1xxx/data/rsdmap/
        public const double Flattening = 5.079304192e-3;

        /// <summary>
        /// Calculate latitude dependent normal gravity of Mars.
        /// References: 
        /// Hofmann-Wellenhof, B. & Moritz, H. (2006), 
        ///  Physical Geodesy, 2nd corr., Springer Wien NewYork. pp. 77-81.
        /// Huber, L. & Beebe, R. (2009), Nasa mars data archive. 
        ///  http://atmos.pds.nasa.gov/data_and_services/atmospheres_data/Mars/Mars.html
        /// </summary>
        /// <param name="latitude">The latitude in degrees.</param>
        /// <returns>The gravity in m/s^2</returns>
        public static double Calculate(double latitude)
        {
            // Model gravity for ellipse
            double latRad = latitude * Math.PI / 180.0;
            double sinLat = Math.Sin(latRad);
            double sin2Lat = Math.Sin(2.0 * latRad);
            double sinLat2 = sinLat * sinLat;
            double sin2Lat2 = sin2Lat * sin2Lat;
            double value = 3.7114219 + 0.02400635 * sinLat2 - 4.2212837e-5 * sin2Lat2;
            return value;
        }
    }
}
