//-----------------------------------------------------------------------------
// MenuEntry.cs
// (Based on XNA Community Game Platform Demo, Microsoft Corp., 2008)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.StateManager
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Helper class represents a single entry in a MenuScreen. By default this
    /// just draws the entry text string, but it can be customized to display menu
    /// entries in different ways. This also provides an event that will be raised
    /// when the menu entry is selected.
    /// </summary>
    public class MenuEntry
    {
        #region Fields

        /// <summary>
        /// The text rendered for this entry.
        /// </summary>
        private string text;

        /// <summary>
        /// Tracks a fading selection effect on the entry.
        /// </summary>
        /// <remarks>
        /// The entries transition out of the selection effect when they are deselected.
        /// </remarks>
        private float selectionFade;

        /// <summary>
        /// Unicolored texture as background pane for text
        /// </summary>
        private Texture2D paneTexture;

        #endregion

        #region Properties

        /// <summary>
        /// Gets or sets the text of this menu entry.
        /// </summary>
        public string Text
        {
            get { return text; }
            set { text = value; }
        }

        #endregion

        #region Events

        /// <summary>
        /// Event raised when the menu entry is selected.
        /// </summary>
        public event EventHandler<PlayerIndexEventArgs> Selected;

        /// <summary>
        /// Method for raising the Selected event.
        /// </summary>
        protected internal virtual void OnSelectEntry(PlayerIndex playerIndex)
        {
            if (Selected != null)
            {
                Selected(this, new PlayerIndexEventArgs(playerIndex));
            }
        }


        #endregion

        #region Initialization


        /// <summary>
        /// Constructs a new menu entry with the specified text.
        /// </summary>
        public MenuEntry(string text)
        {
            this.text = text;
        }


        #endregion

        #region Update and Draw


        /// <summary>
        /// Updates the menu entry.
        /// </summary>
        public virtual void Update(MenuScreen screen, bool isSelected,
                                                      GameTime gameTime)
        {
            // When the menu selection changes, entries gradually fade between
            // their selected and deselected appearance, rather than instantly
            // popping to the new state.
            float fadeSpeed = (float)gameTime.ElapsedGameTime.TotalSeconds * 4;

            if (isSelected)
            {
                selectionFade = Math.Min(selectionFade + fadeSpeed, 1);
            }
            else
            {
                selectionFade = Math.Max(selectionFade - fadeSpeed, 0);
            }
        }


        /// <summary>
        /// Draws the menu entry. This can be overridden to customize the appearance.
        /// </summary>
        public virtual void Draw(MenuScreen screen, Vector2 position,
                                 bool isSelected, GameTime gameTime)
        {
            // Pulsate the size of the selected menu entry.
            double time = gameTime.TotalGameTime.TotalSeconds;
            
            float pulsate = (float)Math.Sin(time * 6) + 1;
            
            float scale = 1 + pulsate * 0.05f * selectionFade;

            // Draw the selected entry in yellow, otherwise white.
            Color textColor = isSelected ? Color.Yellow : Color.White;
            Color paneColor = isSelected ? Color.Yellow : Color.White;

            // Modify the alpha to fade text out during transitions.
            textColor = new Color(
                textColor.R, 
                textColor.G, 
                textColor.B, 
                screen.TransitionAlpha);

            // Modify pane color to be darker
            paneColor = new Color(
                paneColor.R / 64,
                paneColor.G / 64,
                paneColor.B / 64);

            // Create shadow color
            Color color2 = new Color(
                Color.DarkSlateBlue.R,
                Color.DarkSlateBlue.G,
                Color.DarkSlateBlue.B,
                screen.TransitionAlpha);

            // Draw text, centered on the middle of each line.
            ScreenManager screenManager = screen.ScreenManager;
            SpriteBatch spriteBatch = screenManager.SpriteBatch;
            SpriteFont spriteFont = screenManager.Fonts["menu"];

            Vector2 origin = new Vector2(0, spriteFont.LineSpacing / 2);

            // Create translucent pane rectangle 
            if (this.paneTexture == null)
            {
                // New background texture
                this.paneTexture = TextureHelpers.Create(screenManager.GraphicsDevice, new Color(64, 64, 64));
            }

            Vector2 textSize = spriteFont.MeasureString(text);
            Viewport viewport = screenManager.GraphicsDevice.Viewport;
            Rectangle backgroundPane =
                new Rectangle(
                    (int)position.X - 15, 
                    (int)(position.Y - origin.Y + 2), 
                    (int)((viewport.Width / 2) * 1.25f), 
                    screenManager.Fonts["menu"].LineSpacing - 2);
            spriteBatch.Draw(this.paneTexture, backgroundPane, new Color(paneColor, 128));
            if (isSelected)
            {
                backgroundPane.X -= 20;
                backgroundPane.Width = 20;
                spriteBatch.Draw(this.paneTexture, backgroundPane, new Color(paneColor, 200));
            }

            Vector2 shadowPosition;
            Vector2 shadowOffset = new Vector2(2.0f, 2.0f);
            shadowPosition = position + shadowOffset;

            spriteBatch.DrawString(spriteFont, text, shadowPosition, color2, 0,
                       origin, scale, SpriteEffects.None, 0);

            spriteBatch.DrawString(spriteFont, text, position, textColor, 0,
                                   origin, scale, SpriteEffects.None, 0);
        }


        /// <summary>
        /// Queries how much space this menu entry requires.
        /// </summary>
        public virtual int GetHeight(MenuScreen screen)
        {
            return screen.ScreenManager.Fonts["menu"].LineSpacing;
        }


        #endregion
    }
}
