//-----------------------------------------------------------------------------
// MenuScreen.cs
// (Based on XNA Community Game Platform Demo, Microsoft Corp., 2008)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.StateManager
{
    using System;
    using System.Collections.Generic;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Base class for screens that contain a menu of options. The user can
    /// move up and down to select an entry, or cancel to back out of the screen.
    /// </summary>
    public abstract class MenuScreen : GameScreen
    {
        #region Fields

        /// <summary>
        /// The list of menu entries.
        /// </summary>
        private List<MenuEntry> menuEntries = new List<MenuEntry>();

        /// <summary>
        /// The currently selected menu entry.
        /// </summary>
        private int selectedEntry = 0;

        /// <summary>
        /// The title text of the menu.
        /// </summary>
        private string menuTitle;

        /// <summary>
        /// Flag tracking when the screen is uncovered.
        /// </summary>
        private bool waitForUncover = true;

        /// <summary>
        /// Unicolored texture as background pane for text
        /// </summary>
        private Texture2D paneTexture;

        #endregion

        #region Properties

        /// <summary>
        /// Gets the list of menu entries, so derived classes can add
        /// or change the menu contents.
        /// </summary>
        protected IList<MenuEntry> MenuEntries
        {
            get { return this.menuEntries; }
        }

        /// <summary>
        /// Gets or sets the currently selected menu entry.
        /// </summary>
        public int SelectedEntry
        {
            get { return this.selectedEntry; }
            set { this.selectedEntry = Math.Min(value, this.menuEntries.Count); }
        }

        /// <summary>
        /// Gets the list of menu entries, so derived classes can add
        /// or change the menu contents.
        /// </summary>
        protected bool WaitForUncover
        {
            get { return this.waitForUncover; }
            set { this.waitForUncover = value; }
        }

        #endregion

        #region Initialization

        /// <summary>
        /// Constructor.
        /// </summary>
        public MenuScreen(string menuTitle)
        {
            this.menuTitle = menuTitle;

            TransitionOnTime = TimeSpan.FromSeconds(0.5);
            TransitionOffTime = TimeSpan.FromSeconds(0.5);
        }

        #endregion

        #region Handle Input

        /// <summary>
        /// Responds to user input, changing the selected entry and accepting
        /// or cancelling the menu.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            // Move to the previous menu entry?
            if (input.IsNewInputUp(ControllingPlayer))
            {
                this.selectedEntry--;

                if (this.selectedEntry < 0)
                {
                    this.selectedEntry = this.menuEntries.Count - 1;
                }
            }

            // Move to the next menu entry?
            if (input.IsNewInputDown(ControllingPlayer))
            {
                this.selectedEntry++;

                if (this.selectedEntry >= this.menuEntries.Count)
                {
                    this.selectedEntry = 0;
                }
            }

            // Accept or cancel the menu? We pass in our ControllingPlayer, which may
            // either be null (to accept input from any player) or a specific index.
            // If we pass a null controlling player, the InputState helper returns to
            // us which player actually provided the input. We pass that through to
            // OnSelectEntry and OnCancel, so they can tell which player triggered them.
            PlayerIndex playerIndex;

            if (input.IsMenuSelect(ControllingPlayer, out playerIndex))
            {
                OnSelectEntry(this.selectedEntry, playerIndex);
            }
            else if (input.IsMenuCancel(ControllingPlayer, out playerIndex))
            {
                OnCancel(playerIndex);
            }
        }

        /// <summary>
        /// Handler for when the user has chosen a menu entry.
        /// </summary>
        protected virtual void OnSelectEntry(int entryIndex, PlayerIndex playerIndex)
        {
            menuEntries[this.selectedEntry].OnSelectEntry(playerIndex);
        }

        /// <summary>
        /// Handler for when the user has cancelled the menu.
        /// </summary>
        protected virtual void OnCancel(PlayerIndex playerIndex)
        {
            ExitScreen();
        }

        /// <summary>
        /// Helper overload makes it easy to use OnCancel as a MenuEntry event handler.
        /// </summary>
        protected void OnCancel(object sender, PlayerIndexEventArgs e)
        {
            OnCancel(e.PlayerIndex);
        }

        #endregion

        #region Update and Draw

        /// <summary>
        /// Updates the menu.
        /// </summary>
        /// <param name="gameTime"></param>
        /// <param name="coveredByOtherScreen"></param>
        /// <param name="otherScreenHasFocus"></param>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

            // Update each nested MenuEntry object.
            for (int i = 0; i < menuEntries.Count; i++)
            {
                bool isSelected = IsActive && (i == selectedEntry);

                menuEntries[i].Update(this, isSelected, gameTime);
            }
        }


        /// <summary>
        /// Draws the menu.
        /// </summary>
        /// <param name="gameTime"></param>
        public override void Draw(GameTime gameTime)
        {
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            SpriteFont font = ScreenManager.Fonts["menu"];
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;

            // Measure size of menu stack
            float menuSize = 0.0f;
            for (int i = 0; i < menuEntries.Count; i++)
            {
                MenuEntry menuEntry = menuEntries[i];
                menuSize += (menuEntry.GetHeight(this) + 5);
            }

            // Center menu vertically
            Vector2 position = new Vector2(
                viewport.Width / 2 - 200, 
                (viewport.Height - menuSize) / 2);

            // Make the menu slide into place during transitions, using a
            // power curve to make things look more interesting (this makes
            // the movement slow down as it nears the end).
            float transitionOffset = (float)Math.Pow(TransitionPosition, 2);

            if (ScreenState == ScreenState.TransitionOn)
            {
                position.X -= transitionOffset * 256;
            }
            else
            {
                position.X += transitionOffset * 512;
            }

            spriteBatch.Begin();

            // Draw each menu entry in turn.
            for (int i = 0; i < menuEntries.Count; i++)
            {
                MenuEntry menuEntry = menuEntries[i];

                bool isSelected = IsActive && (i == selectedEntry);

                menuEntry.Draw(this, position, isSelected, gameTime);
                

                position.Y += (menuEntry.GetHeight(this) + 5);
            }

            // Create translucent pane rectangle 
            if (this.paneTexture == null)
            {
                // New background texture
                this.paneTexture = TextureHelpers.Create(ScreenManager.GraphicsDevice, new Color(64, 64, 64));
            }

            // Draw the menu title.
            float titleScale = 1.5f;
            Vector2 titlePosition = new Vector2(position.X, viewport.Height/2);
            Vector2 titleSize = font.MeasureString(menuTitle);
            Vector2 titleOrigin = titleSize / 2;
            Color titleColor = new Color(255, 255, 255, TransitionAlpha);
            titlePosition.X -= titleSize.Y * titleScale;
            titlePosition.Y -= transitionOffset * 100;
            Color paneColor = new Color(4, 4, 4);
            Rectangle backgroundPane =
                new Rectangle(
                    (int)(titlePosition.X - titleOrigin.Y), 0, (int)titleSize.Y, viewport.Height);
            spriteBatch.Draw(this.paneTexture, backgroundPane, new Color(paneColor, 128));

            spriteBatch.DrawString(font, menuTitle, titlePosition, titleColor, 3.0f*(float)Math.PI/2.0f,
                                   titleOrigin, titleScale, SpriteEffects.None, 0);

            spriteBatch.End();
        }

        #endregion
    }
}
