//-----------------------------------------------------------------------------
// ScreenManager.cs
// (Based on XNA Community Game Platform Demo, Microsoft Corp., 2008)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.StateManager
{
    using System.Collections.Generic;
    using System.Diagnostics;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Audio;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Media;

    using NewGamePhysics.Utilities;

    /// <summary>
    /// The screen manager is a component which manages one or more GameScreen
    /// instances. It maintains a stack of screens, calls their Update and Draw
    /// methods at the appropriate times, and automatically routes input to the
    /// topmost active screen.
    /// </summary>
    public class ScreenManager : DrawableGameComponent
    {
        #region Fields

        /// <summary>
        /// All screens managed by the ScreenManager.
        /// </summary>
        private List<GameScreen> screens = new List<GameScreen>();

        /// <summary>
        /// All screens that need to be updated.
        /// </summary>
        private List<GameScreen> screensToUpdate = new List<GameScreen>();

        /// <summary>
        /// Input helper.
        /// </summary>
        private InputState input = new InputState();

        /// <summary>
        /// Common sprite batch for all screens.
        /// </summary>
        private SpriteBatch spriteBatch;

        /// <summary>
        /// Common primitive batch for all screens.
        /// </summary>
        private PrimitiveBatch primitiveBatch;

        /// <summary>
        /// Common video player for all screens.
        /// </summary>
        private VideoPlayer videoPlayer;

        /// <summary>
        /// Dictionary of all loaded textures.
        /// </summary>
        private Dictionary<string, Texture2D> textures;

        /// <summary>
        /// Dictionary of all loaded fonts.
        /// </summary>
        private Dictionary<string, SpriteFont> fonts;

        /// <summary>
        /// Dictionary of all loaded models.
        /// </summary>
        private Dictionary<string, Model> models;

        /// <summary>
        /// Dictionary of all loaded sounds.
        /// </summary>
        private Dictionary<string, SoundEffect> sounds;

        /// <summary>
        /// Xact audio engine
        /// </summary>
        private AudioEngine engine;

        /// <summary>
        /// Xact sound bank (cue's).
        /// </summary>
        private SoundBank soundBank;

        /// <summary>
        /// Xact wave bank (wav files).
        /// </summary>
        private WaveBank waveBank;

        /// <summary>
        /// Flag that tracks if the manager was initialized or now.
        /// </summary>
        private bool isInitialized;

        /// <summary>
        /// Flag enabling screen tracing (for debugging purposes).
        /// </summary>
        private bool traceEnabled;

        #endregion

        #region Properties

        /// <summary>
        /// Gets a default SpriteBatch shared by all the screens. This saves
        /// each screen having to bother creating their own local instance.
        /// </summary>
        public SpriteBatch SpriteBatch
        {
            get { return this.spriteBatch; }
        }

        /// <summary>
        /// Gets a default PrimitiveBatch shared by all the screens. This saves
        /// each screen having to bother creating their own local instance.
        /// </summary>
        public PrimitiveBatch PrimitiveBatch
        {
            get { return this.primitiveBatch; }
        }

        /// <summary>
        /// Gets a default VideoPlayer shared by all the screens. This saves
        /// each screen having to bother creating their own local instance.
        /// </summary>
        public VideoPlayer VideoPlayer
        {
            get { return this.videoPlayer; }
        }

        /// <summary>
        /// Gets the texture cache.
        /// </summary>
        public Dictionary<string, Texture2D> Textures
        {
            get { return this.textures; }
        }

        /// <summary>
        /// Gets the font cache.
        /// </summary>
        public Dictionary<string, SpriteFont> Fonts
        {
            get { return this.fonts; }
        }

        /// <summary>
        /// Gets the model cache.
        /// </summary>
        public Dictionary<string, Model> Models
        {
            get { return this.models; }
        }

        /// <summary>
        /// Gets the sound effects cache.
        /// </summary>
        public Dictionary<string, SoundEffect> Sounds
        {
            get { return this.sounds; }
        }

        /// <summary>
        /// Gets the soundbank.
        /// </summary>
        public SoundBank SoundBank
        {
            get { return this.soundBank; }
        }

        /// <summary>
        /// If true, the manager prints out a list of all the screens
        /// each time it is updated. This can be useful for making sure
        /// everything is being added and removed at the right times.
        /// </summary>
        public bool TraceEnabled
        {
            get { return this.traceEnabled; }
            set { this.traceEnabled = value; }
        }

        #endregion

        #region Initialization

        /// <summary>
        /// Constructs a new screen manager component.
        /// </summary>
        public ScreenManager(Game game)
            : base(game)
        {
        }

        /// <summary>
        /// Initializes the screen manager component.
        /// </summary>
        public override void Initialize()
        {
            base.Initialize();

            // Initialize audio objects.
            this.engine = new AudioEngine("Content\\Sounds\\audio.xgs");
            this.soundBank = new SoundBank(engine, "Content\\Sounds\\Sound Bank.xsb");
            this.waveBank = new WaveBank(engine, "Content\\Sounds\\Wave Bank.xwb");

            isInitialized = true;
        }

        /// <summary>
        /// Load your graphics content.
        /// </summary>
        protected override void LoadContent()
        {
            // Load content belonging to the screen manager.
            ContentManager content = Game.Content;
            this.spriteBatch = new SpriteBatch(GraphicsDevice);
            this.primitiveBatch = new PrimitiveBatch(GraphicsDevice);
            this.videoPlayer = new VideoPlayer();

            // Prepare caches
            this.textures = new Dictionary<string, Texture2D>();
            this.fonts = new Dictionary<string, SpriteFont>();
            this.models = new Dictionary<string, Model>();
            this.sounds = new Dictionary<string, SoundEffect>();

            // Prep font cache
            this.fonts.Add("menu", content.Load<SpriteFont>(@"Fonts\menufont"));
            this.fonts.Add("game", content.Load<SpriteFont>(@"Fonts\gamefont"));
            this.fonts.Add("small", content.Load<SpriteFont>(@"Fonts\smallfont"));
            
            // Prep texture cache
            this.textures.Add("blank",content.Load<Texture2D>(@"Sprites\blank"));

            // Tell each of the screens to load their content.
            foreach (GameScreen screen in screens)
            {
                screen.LoadContent();
            }
        }

        /// <summary>
        /// Unload your graphics content.
        /// </summary>
        protected override void UnloadContent()
        {
            // Tell each of the screens to unload their content.
            foreach (GameScreen screen in screens)
            {
                screen.UnloadContent();
            }
        }

        #endregion

        #region Update and Draw

        /// <summary>
        /// Allows each screen to run logic.
        /// </summary>
        public override void Update(GameTime gameTime)
        {
            // Read the keyboard and gamepad.
            this.input.Update();

            // Update the sound engine
            this.engine.Update();

            // Make a copy of the master screen list, to avoid confusion if
            // the process of updating one screen adds or removes others.
            screensToUpdate.Clear();

            foreach (GameScreen screen in screens)
            {
                screensToUpdate.Add(screen);
            }

            bool otherScreenHasFocus = !Game.IsActive;
            bool coveredByOtherScreen = false;

            // Loop as long as there are screens waiting to be updated.
            while (screensToUpdate.Count > 0)
            {
                // Pop the topmost screen off the waiting list.
                GameScreen screen = screensToUpdate[screensToUpdate.Count - 1];

                screensToUpdate.RemoveAt(screensToUpdate.Count - 1);

                // Update the screen.
                screen.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

                if (screen.ScreenState == ScreenState.TransitionOn ||
                    screen.ScreenState == ScreenState.Active)
                {
                    // If this is the first active screen we came across,
                    // give it a chance to handle input.
                    if (!otherScreenHasFocus)
                    {
                        screen.HandleInput(input);

                        otherScreenHasFocus = true;
                    }

                    // If this is an active non-popup, inform any subsequent
                    // screens that they are covered by it.
                    if (!screen.IsPopup)
                    {
                        coveredByOtherScreen = true;
                    }
                }
            }

            // Print debug trace?
            if (traceEnabled)
            {
                TraceScreens();
            }
        }

        /// <summary>
        /// Prints a list of all the screens, for debugging.
        /// </summary>
        void TraceScreens()
        {
            List<string> screenNames = new List<string>();

            foreach (GameScreen screen in screens)
            {
                screenNames.Add(screen.GetType().Name);
            }

            Trace.WriteLine(string.Join(", ", screenNames.ToArray()));
        }

        /// <summary>
        /// Tells each screen to draw itself.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            foreach (GameScreen screen in screens)
            {
                if (screen.ScreenState == ScreenState.Hidden)
                {
                    continue;
                }

                screen.Draw(gameTime);
            }
        }

        #endregion
        
        #region Public Methods

        /// <summary>
        /// Adds a texture to the manager pool
        /// and loads if it has not been loaded yet.
        /// </summary>
        /// <param name="textureKey">The reference key to use for the texture.</param>
        /// <param name="texturePath">The content loader path for the texture.</param>
        public void AddTexture(string textureKey, string texturePath)
        {
            if (!this.textures.ContainsKey(textureKey))
            {
                this.textures.Add(textureKey, Game.Content.Load<Texture2D>(texturePath));
            }
        }

        /// <summary>
        /// Adds a sprite font to the manager pool
        /// and loads if it has not been loaded yet.
        /// </summary>
        /// <param name="fontKey">The reference key to use for the font.</param>
        /// <param name="fontPath">The content loader path for the font.</param>
        public void AddFont(string fontKey, string fontPath)
        {
            if (!this.fonts.ContainsKey(fontKey))
            {
                this.fonts.Add(fontKey, Game.Content.Load<SpriteFont>(fontPath));
            }
        }

        /// <summary>
        /// Adds a sound effect to the manager pool
        /// and loads if it has not been loaded yet.
        /// </summary>
        /// <param name="soundKey">The reference key to use for the sound.</param>
        /// <param name="fontPath">The content loader path for the sound.</param>
        public void AddSound(string soundKey, string soundPath)
        {
            if (!this.sounds.ContainsKey(soundKey))
            {
                this.sounds.Add(soundKey, Game.Content.Load<SoundEffect>(soundPath));
            }
        }

        /// <summary>
        /// Adds a new screen to the screen manager.
        /// </summary>
        public void AddScreen(GameScreen screen, PlayerIndex? controllingPlayer)
        {
            screen.ControllingPlayer = controllingPlayer;
            screen.ScreenManager = this;
            screen.IsExiting = false;

            // If we have a graphics device, tell the screen to load content.
            if (isInitialized)
            {
                screen.LoadContent();
            }

            screens.Add(screen);
        }

        /// <summary>
        /// Removes a screen from the screen manager. You should normally
        /// use GameScreen.ExitScreen instead of calling this directly, so
        /// the screen can gradually transition off rather than just being
        /// instantly removed.
        /// </summary>
        public void RemoveScreen(GameScreen screen)
        {
            // If we have a graphics device, tell the screen to unload content.
            if (isInitialized)
            {
                screen.UnloadContent();
            }

            screens.Remove(screen);
            screensToUpdate.Remove(screen);
        }

        /// <summary>
        /// Expose an array holding all the screens. We return a copy rather
        /// than the real master list, because screens should only ever be added
        /// or removed using the AddScreen and RemoveScreen methods.
        /// </summary>
        public GameScreen[] GetScreens()
        {
            return screens.ToArray();
        }

        /// <summary>
        /// Helper draws a translucent black fullscreen sprite, used for fading
        /// screens in and out, and for darkening the background behind popups.
        /// </summary>
        public void FadeBackBufferToBlack(int alpha)
        {
            Viewport viewport = GraphicsDevice.Viewport;

            spriteBatch.Begin();

            spriteBatch.Draw(this.Textures["blank"],
                             new Rectangle(0, 0, viewport.Width, viewport.Height),
                             new Color(0, 0, 0, (byte)alpha));

            spriteBatch.End();
        }

        /// <summary>
        /// Sets the audio volume for the category.
        /// </summary>
        /// <param name="category">Friendly name of the audio cues.</param>
        /// <param name="volume">The new volume.</param>
        public void SetVolume(string category, float volume)
        {
            AudioCategory musicCategory = this.engine.GetCategory(category);
            if (null != musicCategory)
            {
                musicCategory.SetVolume(volume);
            }
        }

        #endregion
    }
}
