

namespace NewGamePhysics.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Text;

    using Microsoft.DirectX.DirectInput;

    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Input;

    public class Gamepads
    {
		private List<DirectXInput> items;

		public List<DirectXInput> Items
		{
			get
			{
				return items;
			}
		}
		/// <summary>
		/// Reloads all gamepads
        /// </summary>
        public Gamepads()
		{
			// gamepads generally misidentified as Joysticks in DirectInput... get both
			DeviceList gamepadInstanceList = Manager.GetDevices(DeviceType.Gamepad, EnumDevicesFlags.AttachedOnly);
			DeviceList joystickInstanceList = Manager.GetDevices(DeviceType.Joystick, EnumDevicesFlags.AttachedOnly);

			items = new List<DirectXInput>(gamepadInstanceList.Count + joystickInstanceList.Count);

			foreach (DeviceInstance deviceInstance in gamepadInstanceList)
			{
				DirectXInput item = new DirectXInput(deviceInstance.InstanceGuid);
				items.Add(item);
			}

			foreach (DeviceInstance deviceInstance in joystickInstanceList)
			{
				DirectXInput item = new DirectXInput(deviceInstance.InstanceGuid);
				items.Add(item);
			}
		}

    }

    public class DirectXInput
    {
		protected Device device;
		public Device Device
		{
			get { return device; }
		}

		public DirectXInput(Guid gamepadInstanceGuid)
		{
			device = new Device(gamepadInstanceGuid);
			device.SetDataFormat(DeviceDataFormat.Joystick);
			device.Acquire();			
		}

		public DirectInputThumbSticks ThumbSticks
		{
			get { return new DirectInputThumbSticks(device); }
		}

		public DirectInputDPad DPad
		{
			get
			{
				JoystickState t = device.CurrentJoystickState;
				return new DirectInputDPad(t.GetPointOfView()[0]);	
                // note that there could be a total of 4 DPads on the PC
			}
		}

		public DirectInputButtons Buttons
		{
			get { return new DirectInputButtons(device); }
		}

		#region Diagnostics

		public string DiagnosticsThumbSticks
		{
			get
			{
				return
					"X" + Math.Round(ThumbSticks.Left.X, 4) +
					" Y" + Math.Round(ThumbSticks.Left.Y, 4) +
					" X" + Math.Round(ThumbSticks.Right.X, 4) +
					" Y" + Math.Round(ThumbSticks.Right.Y, 4);
			}
		}

		public string DiagnosticsRawGamepadData
		{
			get
			{
				return
					"X" + Device.CurrentJoystickState.X +
					" Y" + Device.CurrentJoystickState.Y +
					" Z" + Device.CurrentJoystickState.Z +
					" Rx" + Device.CurrentJoystickState.Rx +
					" Ry" + Device.CurrentJoystickState.Ry +
					" Rz" + Device.CurrentJoystickState.Rz +
					" pov[0]" + Device.CurrentJoystickState.GetPointOfView()[0];
			}
		}

		public string DiagnosticsButtons
		{
			get
			{
				System.Text.StringBuilder sb = new System.Text.StringBuilder();

				int i = 0;
				foreach (ButtonState bs in Buttons.List)
				{
					sb.Append(i);
					sb.Append("=");
					sb.Append((bs == ButtonState.Pressed ? "1" : "0"));
					sb.Append(" ");
					i++;
				}

				return sb.ToString();
			}
		}

		#endregion

    }

    /// <summary>
    /// A struct offering the current positions of as many as 3 thumbsticks on a PC gamepad or joystick
    /// </summary>
    /// <remarks>
    /// For unusual joysticks, these "thumbsticks" may be whatever the hardware-designer imagined;
    /// for example, Right.Y might be a jet-throttle and Right.X might be the rotational position of a steering wheel
    /// In other words, being in the list of Gamepads doesn't mean it looks anything like a Gamepad
    /// </remarks>
    public struct DirectInputThumbSticks
    {
        /// <summary>
        /// Check HasLeft, HasRight, etc before getting these values; 
        /// will always be 0 if this gamepad lacks the requested thumbstick
        /// </summary>
        public Vector2 Left;
        public Vector2 Right;
        public Vector2 Third;
        //public Microsoft.Xna.Framework.Vector2 Fourth;

        public bool HasLeft;
        public bool HasRight;
        public bool HasThird;
        //public bool HasFourth;

        const float center = 32767.5f;

        public DirectInputThumbSticks(Device device)
        {
            JoystickState t = device.CurrentJoystickState;

            HasLeft = false;
            Left = Vector2.Zero;
            HasRight = false;
            Right = Vector2.Zero;
            HasThird = false;
            Third = Vector2.Zero;

            if (device.Caps.NumberAxes > 0)
            {
                HasLeft = true;
                Left = new Vector2((t.X - center) / center, (t.Y - center) / center);

                if (device.Caps.NumberAxes > 2)
                {
                    HasRight = true;
                    Right = new Vector2((t.Rz - center) / center, (t.Z - center) / center);

                    if (device.Caps.NumberAxes > 4)
                    {
                        HasThird = true;
                        Third = new Vector2((t.Rx - center) / center, (t.Ry - center) / center);
                    }
                }
            }
        }
    }

    public struct DirectInputDPad
    {
        public ButtonState Up;
        public ButtonState Right;
        public ButtonState Down;
        public ButtonState Left;

        public DirectInputDPad(int direction)
        {
            Up = ButtonState.Released;
            Right = ButtonState.Released;
            Down = ButtonState.Released;
            Left = ButtonState.Released;

            if (direction == -1)
                return;

            if (direction > 27000 || direction < 9000)
            {
                Up = ButtonState.Pressed;
            }

            if (0 < direction && direction < 18000)
            {
                Right = ButtonState.Pressed;
            }

            if (9000 < direction && direction < 27000)
            {
                Down = ButtonState.Pressed;
            }

            if (18000 < direction)
            {
                Left = ButtonState.Pressed;
            }
        }
    }

    /// <summary>
    /// The buttons on a PC Gamepad
    /// </summary>
    /// <remarks>
    /// The comparable X, Y, A, B etc are not currently mapped because the position of buttons
    /// indexed 0, 1, 2, etc varies widely from PC gamepad to PC gamepad
    /// Instead, you should provide an interface for the user to visually map the buttons on their
    /// gamepad to what you expect for X, Y, etc.
    /// Tools to support this mapping may be added to this framework in the future
    /// </remarks>
    public struct DirectInputButtons
    {
        /*
        public ButtonState X;
        public ButtonState Y;
        public ButtonState A;
        public ButtonState B;
        public ButtonState Back;
        public ButtonState Start;
        public ButtonState LeftShoulder;
        public ButtonState RightShoulder;
        public ButtonState LeftStick;
        public ButtonState RightStick;
        */

        public List<ButtonState> List;

        public DirectInputButtons(Device device)
        {
            byte[] buttons = device.CurrentJoystickState.GetButtons();

            /*
            X = (buttons[0] == 0 ? ButtonState.Released : ButtonState.Pressed);
            Y = (buttons[1] == 0 ? ButtonState.Released : ButtonState.Pressed);
            A = (buttons[2] == 0 ? ButtonState.Released : ButtonState.Pressed);
            B = (buttons[3] == 0 ? ButtonState.Released : ButtonState.Pressed);
            Back = (buttons[4] == 0 ? ButtonState.Released : ButtonState.Pressed);
            Start = (buttons[5] == 0 ? ButtonState.Released : ButtonState.Pressed);
            LeftShoulder = (buttons[6] == 0 ? ButtonState.Released : ButtonState.Pressed);
            RightShoulder = (buttons[7] == 0 ? ButtonState.Released : ButtonState.Pressed);
            LeftStick = (buttons[8] == 0 ? ButtonState.Released : ButtonState.Pressed);
            RightStick = (buttons[9] == 0 ? ButtonState.Released : ButtonState.Pressed);
            */

            int numButtons = device.Caps.NumberButtons;
            List = new List<ButtonState>(numButtons);

            for (int i = 0; i < numButtons; i++)
            {
                List.Add((buttons[i] == 0 ? ButtonState.Released : ButtonState.Pressed));
            }
        }
    }
}
