﻿#region File Description
//-----------------------------------------------------------------------------
// NativeAudio.cs
//-----------------------------------------------------------------------------
#endregion

namespace NewGamePhysics.Utilities
{
    using System;
    using System.Threading;
    using System.Runtime.InteropServices;

    /// <summary>
    /// Wave format definition for unmanaged interop.
    /// </summary>
    [StructLayout(LayoutKind.Sequential)] 
    public class WaveFormat
    {
        /// <summary>
        /// The format tag. Locked to PCM in constructor. 
        /// </summary>
        public ushort formatTag;

        /// <summary>
        /// Number of channels (1 or 2).
        /// </summary>
        public ushort channels;

        /// <summary>
        /// Samples per second (>0).
        /// </summary>
        public uint samplesPerSec;

        /// <summary>
        /// Average bytes per second.
        /// </summary>
        public uint averageBytesPerSec;

        /// <summary>
        /// Block alignment.
        /// </summary>
        public ushort blockAlignment;

        /// <summary>
        /// Bits per sample (8 or 16).
        /// </summary>
        public ushort bitsPerSample;

        /// <summary>
        /// Filler short.
        /// </summary>
        public short cbSize;

        /// <summary>
        /// Creates a wave format object used to initialize the native audio interface.
        /// </summary>
        /// <param name="rate">Playback rate in samples per second (>0).</param>
        /// <param name="bits">Bits per sample (8 or 16).</param>
        /// <param name="channels">Number of channels (1 or 2).</param>
        public WaveFormat(uint rate, ushort bits, ushort channels)
        {
            if (!(rate > 0))
            {
                throw new ArgumentException("'rate' must be greater than >0", "rate");
            }

            if (!((bits == 8) || (bits == 16)))
            {
                throw new ArgumentException("'bits' must be 8 or 16", "bits");
            }

            if (!((channels == 1) || (channels == 2)))
            {
                throw new ArgumentException("'channels' must be 1 or 2", "channels");
            }

            this.formatTag = 1; // lock to PCM
            this.channels = channels;
            this.samplesPerSec = rate;
            this.bitsPerSample = bits;
            this.cbSize = 0;
               
            this.blockAlignment = (ushort)(channels * (bits / 8));
            this.averageBytesPerSec = samplesPerSec * blockAlignment;
        }
    }

    /// <summary>
    /// Implements an interface for native audio playback 
    /// </summary>
    public class NativeAudioInterface
    {
        /// <summary>
        /// Base for MM errors.
        /// </summary>
        private const int MMSYSERR_BASE = 0;

        /// <summary>
        /// MM errors enumeration.
        /// </summary>
        private enum MMSYSERR : int
        {
            NOERROR = 0,
            ERROR = (MMSYSERR_BASE + 1),
            BADDEVICEID = (MMSYSERR_BASE + 2),
            NOTENABLED = (MMSYSERR_BASE + 3),
            ALLOCATED = (MMSYSERR_BASE + 4),
            INVALHANDLE = (MMSYSERR_BASE + 5),
            NODRIVER = (MMSYSERR_BASE + 6),
            NOMEM = (MMSYSERR_BASE + 7),
            NOTSUPPORTED = (MMSYSERR_BASE + 8),
            BADERRNUM = (MMSYSERR_BASE + 9),
            INVALFLAG = (MMSYSERR_BASE + 10),
            INVALPARAM = (MMSYSERR_BASE + 11),
            HANDLEBUSY = (MMSYSERR_BASE + 12),
            INVALIDALIAS = (MMSYSERR_BASE + 13),
            BADDB = (MMSYSERR_BASE + 14),
            KEYNOTFOUND = (MMSYSERR_BASE + 15),
            READERROR = (MMSYSERR_BASE + 16),
            WRITEERROR = (MMSYSERR_BASE + 17),
            DELETEERROR = (MMSYSERR_BASE + 18),
            VALNOTFOUND = (MMSYSERR_BASE + 19),
            NODRIVERCB = (MMSYSERR_BASE + 20),
            LASTERROR = (MMSYSERR_BASE + 20)
        }

        /// <summary>
        /// Base for wave errors.
        /// </summary>
        private const int WAVERR_BASE = 32;

        /// <summary>
        /// Wave error enumerations.
        /// </summary>
        private enum WAVERR : int
        {
            NONE = 0,
            BADFORMAT = WAVERR_BASE + 0,
            STILLPLAYING = WAVERR_BASE + 1,
            UNPREPARED = WAVERR_BASE + 2,
            SYNC = WAVERR_BASE + 3,
            LASTERROR = WAVERR_BASE + 3
        }

        /// <summary>
        /// Constant indicating no error from the MM system.
        /// </summary>
        public const int MMSYSERR_NOERROR = 0;

        /// <summary>
        /// Constant indicating MM system open state.
        /// </summary>
        public const int MM_WOM_OPEN = 0x3BB;

        /// <summary>
        /// Constant indicating MM system close state.
        /// </summary>
        public const int MM_WOM_CLOSE = 0x3BC;

        /// <summary>
        ///  Constant indicating MM system done state.
        /// </summary>
        public const int MM_WOM_DONE = 0x3BD;

        /// <summary>
        /// Constant indicating dwCallback is a FARPROC.
        /// </summary>
        public const int CALLBACK_FUNCTION = 0x00030000;

        /// <summary>
        /// Callback delegate definition.
        /// </summary>
        /// <param name="hdrvr">Pointer to wave header.</param>
        /// <param name="uMsg"></param>
        /// <param name="dwUser">Client parameter.</param>
        /// <param name="wavhdr">Header structure.</param>
        /// <param name="dwParam2"></param>
        public delegate void WaveDelegate(
            IntPtr hdrvr, 
            int uMsg, 
            int dwUser, 
            ref WaveHeader wavhdr, 
            int dwParam2);

        /// <summary>
        /// Wave header structure for native interop.
        /// </summary>
        [StructLayout(LayoutKind.Sequential)] public struct WaveHeader
        {
            /// <summary>
            /// Pointer to locked data buffer.
            /// </summary>
            public IntPtr lpData;

            /// <summary>
            /// Length of data buffer.
            /// </summary>
            public int dwBufferLength;

            /// <summary>
            /// Unused (i.e. used for input only).
            /// </summary>
            public int dwBytesRecorded;

            /// <summary>
            /// Client parameter.
            /// </summary>
            public IntPtr dwUser;

            /// <summary>
            /// Various flags.
            /// </summary>
            public int dwFlags;

            /// <summary>
            /// Loop control counter.
            /// </summary>
            public int dwLoops;

            /// <summary>
            /// WaveHdr, reserved for driver.
            /// </summary>
            public IntPtr lpNext;

            /// <summary>
            /// Reserved for driver.
            /// </summary>
            public int reserved;
        }

        #region native_calls
        /// <summary>
        /// Native dll to use.
        /// </summary>
        private const string mmdll = "winmm.dll";

        [DllImport(mmdll)]
        public static extern int waveOutPrepareHeader(IntPtr hWaveOut, ref WaveHeader lpWaveOutHdr, int uSize);
        
        [DllImport(mmdll)]
        public static extern int waveOutUnprepareHeader(IntPtr hWaveOut, ref WaveHeader lpWaveOutHdr, int uSize);
        
        [DllImport(mmdll)]
        public static extern int waveOutWrite(IntPtr hWaveOut, ref WaveHeader lpWaveOutHdr, int uSize);
        
        [DllImport(mmdll)]
        public static extern int waveOutOpen(out IntPtr hWaveOut, int uDeviceID, WaveFormat lpFormat, WaveDelegate dwCallback, int dwInstance, int dwFlags);
        
        [DllImport(mmdll)]
        public static extern int waveOutReset(IntPtr hWaveOut);
        
        [DllImport(mmdll)]
        public static extern int waveOutClose(IntPtr hWaveOut);

        #endregion
    }

    /// <summary>
    /// Delegate signature to fill a new data buffer with samples.
    /// </summary>
    /// <param name="data"></param>
    public delegate void BufferFillCallback(byte[] data);

    /// <summary>
    /// Object which represents a data buffer of audio samples.
    /// </summary>
	internal class SampleBuffer : IDisposable
	{
        /// <summary>
        ///  Maximum number of seconds to wait for a sample buffer to finish playing.
        /// </summary>
        private const int BufferWaitTimeoutSeconds = 10;

        /// <summary>
        /// Pointer to data buffer for playback.
        /// </summary>
        private IntPtr waveData;

        /// <summary>
        /// Event tracking if a buffer has completed playing.
        /// </summary>
        private AutoResetEvent bufferCompleted = new AutoResetEvent(false);

        /// <summary>
        /// Current wave header.
        /// </summary>
        private NativeAudioInterface.WaveHeader header;

        /// <summary>
        /// Handle to wave header.
        /// </summary>
        private GCHandle headerHandle;

        /// <summary>
        /// Current wave data.
        /// </summary>
        private byte[] headerData;

        /// <summary>
        /// Handle to wave data.
        /// </summary>
        private GCHandle headerDataHandle;

        /// <summary>
        /// State flag to track if buffers are playing.
        /// </summary>
        private bool isPlaying;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="hdrvr"></param>
        /// <param name="message"></param>
        /// <param name="dwUser"></param>
        /// <param name="waveHeader"></param>
        /// <param name="dwParam2"></param>
        internal static void WaveOutProc(
            IntPtr hdrvr, 
            int message, 
            int dwUser, 
            ref NativeAudioInterface.WaveHeader waveHeader, 
            int dwParam2)
        {
            if (message == NativeAudioInterface.MM_WOM_DONE)
            {
                try
                {
                    GCHandle h = (GCHandle)waveHeader.dwUser;
                    SampleBuffer buffer = (SampleBuffer)h.Target;
                    buffer.OnCompleted();
                }
                catch
                {
                }
            }
        }

        /// <summary>
        /// Represents a sample buffer which can be played via the native interface.
        /// </summary>
        /// <param name="waveOutHandle">Pointer to sample buffer.</param>
        /// <param name="bufferSize">Size of sample buffer.</param>
        public SampleBuffer(IntPtr waveOutHandle, int bufferSize)
		{
            // Sample buffer
            waveData = waveOutHandle;

            // Create header
            headerHandle = GCHandle.Alloc(header, GCHandleType.Pinned);
            header.dwUser = (IntPtr)GCHandle.Alloc(this);

            // Create sample buffer
            headerData = new byte[bufferSize];
            headerDataHandle = GCHandle.Alloc(headerData, GCHandleType.Pinned);
            header.lpData = headerDataHandle.AddrOfPinnedObject();
            header.dwBufferLength = bufferSize;

            // Prepare sample buffer
            int error = NativeAudioInterface.waveOutPrepareHeader(
                waveData, 
                ref header, 
                Marshal.SizeOf(header));
            if (error != NativeAudioInterface.MMSYSERR_NOERROR)
            {
                throw new ApplicationException("Could not prepare sample buffer.");
            }
		}

        /// <summary>
        /// Destructor for sample buffer.
        /// </summary>
        ~SampleBuffer()
        {
            Dispose();
        }

        /// <summary>
        /// Dispose of sample buffer.
        /// </summary>
        public void Dispose()
        {
            // Clean header
            if (header.lpData != IntPtr.Zero)
            {
                NativeAudioInterface.waveOutUnprepareHeader(waveData, ref header, Marshal.SizeOf(header));
                headerHandle.Free();
                header.lpData = IntPtr.Zero;
            }
            
            // Clean sample buffer
            if (headerDataHandle.IsAllocated)
            {
                headerDataHandle.Free();
            }

            // Clean event
            bufferCompleted.Close();

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// Gets the size of the sample buffer.
        /// </summary>
        public int Size
        {
            get { return header.dwBufferLength; }
        }

        /// <summary>
        /// Gets a pointer to the sample buffer.
        /// </summary>
        public IntPtr Data
        {
            get { return header.lpData; }
        }

        /// <summary>
        /// Start playing the sample buffer.
        /// </summary>
        /// <returns>True if playback was be started.</returns>
        public bool Play()
        {
            lock(this)
            {
                this.bufferCompleted.Reset();
                int result = NativeAudioInterface.waveOutWrite(
                    waveData, 
                    ref header, 
                    Marshal.SizeOf(header));
                this.isPlaying = (result == NativeAudioInterface.MMSYSERR_NOERROR);                
            }

            return this.isPlaying;
        }

        /// <summary>
        /// Waits for event indicating that a buffer has completed playing.
        /// </summary>
        public void WaitForBuffer()
        {
            if (this.isPlaying)
            {
                this.bufferCompleted.WaitOne(new TimeSpan(0, 0, BufferWaitTimeoutSeconds));
            }
        }

        /// <summary>
        /// Set event that buffer has completed playing.
        /// </summary>
        public void OnCompleted()
        {
            lock (this)
            {
                this.bufferCompleted.Set();
                this.isPlaying = false;
            }
        }
    }

    /// <summary>
    /// Plays audio via the waveOut device.
    /// </summary>
    public class WaveOutPlayer : IDisposable
    {
        /// <summary>
        /// Number of buffers.
        /// </summary>
        private const int NumBuffers = 2;

        /// <summary>
        /// Number of bytes in buffer (about 0.1 second in 44khz/stereo/16bit).
        /// </summary>
        private const int BufferSize = 4096;

        /// <summary>
        /// Pointer to sample buffer.
        /// </summary>
        private IntPtr waveOutHandle;

        /// <summary>
        /// Array of all sample buffers.
        /// </summary>
        private SampleBuffer[] buffers = new SampleBuffer[NumBuffers];

        /// <summary>
        /// Current sample buffer to be processed.
        /// </summary>
        private int currentBuffer;

        /// <summary>
        /// Thread which fills sample buffers with data.
        /// </summary>
        private Thread fillBufferThread;

        /// <summary>
        /// Callback routine which provides sample data.
        /// </summary>
        private BufferFillCallback bufferFillCallback;

        /// <summary>
        /// Flag indicating we are done playing.
        /// </summary>
		private bool finished;

        /// <summary>
        /// The value for a silent (zero) sample.
        /// </summary>
		private byte zeroSampleValue;

        /// <summary>
        /// Callback to send sample buffer to audio device.
        /// </summary>
        private NativeAudioInterface.WaveDelegate waveOutProc = 
            new NativeAudioInterface.WaveDelegate(SampleBuffer.WaveOutProc);

        /// <summary>
        /// Plays audio via waveOut device.
        /// </summary>
        /// <param name="format">Playback format to use.</param>
        /// <param name="bufferFillCallback">The callback delegate which provides sample data.</param>
        public WaveOutPlayer( 
            WaveFormat format, 
            BufferFillCallback bufferFillCallback)
        {
            // Determine quiet sample value
			this.zeroSampleValue = format.bitsPerSample == 8 ? (byte)128 : (byte)0;

            // Remember callback
            this.bufferFillCallback = bufferFillCallback;

            // Open wave out device.
            int result = NativeAudioInterface.waveOutOpen(
                out waveOutHandle, 
                -1, 
                format, 
                waveOutProc,
                0, 
                NativeAudioInterface.CALLBACK_FUNCTION);
            if (result != NativeAudioInterface.MMSYSERR_NOERROR)
            {
                throw new ApplicationException("Could not open wave out device. Error: " + result);
            }

            // Allocate sample buffers.
            AllocateBuffers();

            // Start thread which provides sample data.
            fillBufferThread = new Thread(new ThreadStart(WaveOutPlayerThread));
            fillBufferThread.Start();
        }

        /// <summary>
        /// Destructor for the player.
        /// </summary>
        ~WaveOutPlayer()
        {
            Dispose();
        }

        /// <summary>
        /// Dispose of waveOut player.
        /// </summary>
        public void Dispose()
        {
            if (fillBufferThread != null)
            {
                try
                {
                    finished = true;
                    if (waveOutHandle != IntPtr.Zero)
                    {
                        NativeAudioInterface.waveOutReset(waveOutHandle);
                    }

                    fillBufferThread.Join();
                    bufferFillCallback = null;
                    FreeBuffers();
                    if (waveOutHandle != IntPtr.Zero)
                    {
                        NativeAudioInterface.waveOutClose(waveOutHandle);
                    }
                }
                finally
                {
                    fillBufferThread = null;
                    waveOutHandle = IntPtr.Zero;
                }
            }

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// Thread which fills sample buffers with data.
        /// </summary>
        private void WaveOutPlayerThread()
        {
            while (!finished)
            {
                byte[] sampleData = new byte[BufferSize];
                
                if (bufferFillCallback != null)
                {
                    // Fill samples via callback
                    bufferFillCallback(sampleData);
                }
                else
                {
                    // Create a buffer with zero sample values
                    for (int i = 0; i < BufferSize; i++)
                    {
                        sampleData[i] = zeroSampleValue;
                    }
                }

                // Wait for buffer to complete playing
                this.buffers[currentBuffer].WaitForBuffer();

                // Copy new sample data to unmanaged side
                System.Runtime.InteropServices.Marshal.Copy(
                    sampleData, 0, buffers[currentBuffer].Data, buffers[currentBuffer].Size);

                // Trigger playback of buffer via wavOut device
                this.buffers[currentBuffer].Play();

                // Advance to next buffer to fill
                currentBuffer++;
                currentBuffer = currentBuffer % NumBuffers;                 
            }

            // Wait for all buffers to finish playing
			WaitForAllBuffers();
		}

        /// <summary>
        /// Create all sample buffers.
        /// </summary>
        private void AllocateBuffers()
        {
            FreeBuffers();
            for (int i = 0; i < NumBuffers; i++)
            {
                this.buffers[i] = new SampleBuffer(this.waveOutHandle, BufferSize);
            }
        }

        /// <summary>
        /// Dispose of all sample buffers.
        /// </summary>
        private void FreeBuffers()
        {
            for (int i = 0; i < NumBuffers; i++)
            {
                if (this.buffers[i] != null)
                {
                    this.buffers[i].Dispose();
                    this.buffers[i] = null;
                }
            }
        }

        /// <summary>
        /// Wait for all buffers to finish playing.
        /// </summary>
        private void WaitForAllBuffers()
        {
            for (int i = 0; i < NumBuffers; i++)
            {
                if (this.buffers[i] != null)
                {
                    this.buffers[i].WaitForBuffer();
                }
            }
        }
    }
}

