﻿//-----------------------------------------------------------------------------
// Scanf.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace NewGamePhysics.Utilities
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Globalization;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;

    /// <summary>
    /// Enumeration of possible parse types.
    /// </summary>
    internal enum ParseType
    {
        Null,
        Int16,
        UInt16,
        UInt16Oct,
        UInt16Hex,
        Int32,
        UInt32,
        UInt32Oct,
        UInt32Hex,
        Int64,
        UInt64,
        UInt64Oct,
        UInt64Hex,
        Single,
        String,
        Double,
        Char
    }

    /// <summary>
    /// C# sscanf simulator.
    /// Original idea from Scanner.cs by Geoffrey Slinker
    /// </summary>
    public class Scanf
    {
        /// <summary>
        /// Regexp which matches an integer number.
        /// </summary>
        private const string integerPattern = @"[-+]?[0-9]+";

        /// <summary>
        /// Regexp which matches an unsigned octal number.
        /// </summary>
        private const string octalIntegerPattern = @"[0-7]+";

        /// <summary>
        /// Regexp which matches an unsigned hexadecimal number.
        /// </summary>
        private const string hexadecimalIntegerPattern = @"[0-9a-fA-F]+";

        /// <summary>
        /// Regexp which matches a floating point number.
        /// </summary>
        private const string floatingPointPattern = @"[-+]?[0-9]*\.?[0-9]+([eE][-+]?[0-9]+)?";

        /// <summary>
        /// Regexp which matches a non-whitespace string.
        /// </summary>
        private const string stringPattern = @"[\w\d\S]+";

        /// <summary>
        /// Regexp which matches a single character.
        /// </summary>
        private const string characterPattern = @"[\w\S]{1}";

        /// <summary>
        /// Seperator between parse type string and match group index.
        /// </summary>
        private static char[] seperator = { '_' };

        /// <summary>
        /// List of scanf patterns with corresponding Regexp strings.
        /// </summary>
        protected readonly Hashtable patterns;

        /// <summary>
        /// List of types for each scanf pattern. 
        /// </summary>
        protected readonly Hashtable types;

        /// <summary>
        /// Enables trace logging when set to true. 
        /// Used for debugging.
        /// </summary>
        public static bool EnableTrace = false;

        /// <summary>
        /// The match evaluator for the pattern conversion.
        /// </summary>
        private MatchEvaluator matchEvaluator;

        /// <summary>
        /// Tracks the current type index to create unique names in the
        /// match callback.
        /// </summary>
        private int typeCount;

        /// <summary>
        /// Remember last processed format in this cache
        /// to determine if the regular expression needs to be rebuild.
        /// </summary>
        private string lastFormat;

        /// <summary>
        /// The current conversion regular expression.
        /// </summary>
        private Regex conversionRegex;

        /// <summary>
        /// Constructor for Scanf object. 
        /// Initializes the pattern tables.
        /// </summary>
        public Scanf()
        {
            // New format-to-regexp pattern table
            patterns = new Hashtable();

            // base-types
            patterns.Add("%d", integerPattern);
            patterns.Add("%i", integerPattern);
            patterns.Add("%o", octalIntegerPattern);
            patterns.Add("%u", integerPattern);
            patterns.Add("%x", hexadecimalIntegerPattern);
            patterns.Add("%X", hexadecimalIntegerPattern);
            patterns.Add("%f", floatingPointPattern);
            patterns.Add("%g", floatingPointPattern);
            patterns.Add("%G", floatingPointPattern);
            patterns.Add("%s", stringPattern);
            patterns.Add("%c", characterPattern);

            // long-types
            patterns.Add("%lf", floatingPointPattern);
            patterns.Add("%lg", floatingPointPattern);
            patterns.Add("%lG", floatingPointPattern);
            patterns.Add("%ld", integerPattern);
            patterns.Add("%li", integerPattern);
            patterns.Add("%lo", octalIntegerPattern);
            patterns.Add("%lu", integerPattern);
            patterns.Add("%lx", hexadecimalIntegerPattern);
            patterns.Add("%lX", hexadecimalIntegerPattern);

            // short-types
            patterns.Add("%hd", integerPattern);
            patterns.Add("%hi", integerPattern);
            patterns.Add("%ho", octalIntegerPattern);
            patterns.Add("%hu", integerPattern);
            patterns.Add("%hx", hexadecimalIntegerPattern);
            patterns.Add("%hX", hexadecimalIntegerPattern);

            // Skips
            patterns.Add(@"%\*d", integerPattern);
            patterns.Add(@"%\*i", integerPattern);
            patterns.Add(@"%\*o", octalIntegerPattern);
            patterns.Add(@"%\*u", integerPattern);
            patterns.Add(@"%\*x", hexadecimalIntegerPattern);
            patterns.Add(@"%\*X", hexadecimalIntegerPattern);
            patterns.Add(@"%\*f", floatingPointPattern);
            patterns.Add(@"%\*g", floatingPointPattern);
            patterns.Add(@"%\*G", floatingPointPattern);
            patterns.Add(@"%\*s", stringPattern);
            patterns.Add(@"%\*c", characterPattern);

            patterns.Add(@"%\*lf", floatingPointPattern);
            patterns.Add(@"%\*lg", floatingPointPattern);
            patterns.Add(@"%\*lG", floatingPointPattern);
            patterns.Add(@"%\*ld", integerPattern);
            patterns.Add(@"%\*li", integerPattern);
            patterns.Add(@"%\*lo", octalIntegerPattern);
            patterns.Add(@"%\*lu", integerPattern);
            patterns.Add(@"%\*lx", hexadecimalIntegerPattern);
            patterns.Add(@"%\*lX", hexadecimalIntegerPattern);

            patterns.Add(@"%\*hd", integerPattern);
            patterns.Add(@"%\*hi", integerPattern);
            patterns.Add(@"%\*ho", octalIntegerPattern);
            patterns.Add(@"%\*hu", integerPattern);
            patterns.Add(@"%\*hx", hexadecimalIntegerPattern);
            patterns.Add(@"%\*hX", hexadecimalIntegerPattern);

            // New format-to-parse type table
            types = new Hashtable();

            types.Add("%d", ParseType.Int32);
            types.Add("%i", ParseType.Int32);
            types.Add("%o", ParseType.UInt32Oct);
            types.Add("%u", ParseType.UInt32);
            types.Add("%x", ParseType.UInt32Hex);
            types.Add("%X", ParseType.UInt32Hex);
            types.Add("%f", ParseType.Single);
            types.Add("%g", ParseType.Single);
            types.Add("%G", ParseType.Single);
            types.Add("%s", ParseType.String);
            types.Add("%c", ParseType.Char);

            types.Add("%lf", ParseType.Double);
            types.Add("%lg", ParseType.Double);
            types.Add("%lG", ParseType.Double);
            types.Add("%ld", ParseType.Int64);
            types.Add("%li", ParseType.Int64);
            types.Add("%lo", ParseType.UInt64Oct);
            types.Add("%lu", ParseType.UInt64);
            types.Add("%lx", ParseType.UInt64Hex);
            types.Add("%lX", ParseType.UInt64Hex);

            types.Add("%hd", ParseType.Int16);
            types.Add("%hi", ParseType.Int16);
            types.Add("%ho", ParseType.UInt16Oct);
            types.Add("%hu", ParseType.UInt16);
            types.Add("%hx", ParseType.UInt16Hex);
            types.Add("%hX", ParseType.UInt16Hex);

            // New match evaluator
            matchEvaluator = new MatchEvaluator(ReplaceFormatCode);
        }

        /// <summary>
        /// Parses a text using a scanf-like format representation into 
        /// an array of objects.
        /// </summary>
        /// <param name="text">
        /// Test to parse.
        /// </param>
        /// <param name="format">
        /// Format string to use for parsing.
        /// </param>
        /// <returns>
        /// Array of objects from parse. 
        /// One element for each recognized pattern.
        /// null elements for skipped patterns.
        /// </returns>
        public object[] Scan(string text, string format)
        {
            // Check if we need to update our regular expression for the format
            if ((String.IsNullOrEmpty(lastFormat)) ||
                (lastFormat != format))
            {
                // Convert format string into pattern
                string formatPattern = format;

                // Multiple whitespaces reduce to single whitespace match
                formatPattern = Regex.Replace(formatPattern, @"\s+", @"\s+");

                // Type names are individualized with a counter 
                // The count is updated inside the match evaluator.
                typeCount = 0;

                // Maybe rebuild format pattern
                foreach (string formatCode in patterns.Keys)
                {
                    formatPattern = Regex.Replace(
                        formatPattern, 
                        formatCode, 
                        matchEvaluator);
                }
                Trace("scanf_trace: format_pattern: ", formatPattern);

                // New regular expression for format
                conversionRegex = new Regex(formatPattern);

                // Cache format for reuse tracking
                lastFormat = format;
            }

            // Get the array of group names
            string[] groupNames = conversionRegex.GetGroupNames();

            // Initalize return array
            List<object> targets = new List<object>();

            if (groupNames.Length > 0)
            {
                // Match pattern
                MatchCollection matchCollection = conversionRegex.Matches(text);
                foreach (Match match in matchCollection)
                {
                    if (match.Success)
                    {
                        foreach (string groupName in groupNames)
                        {
                            if (groupName.Contains(seperator[0]))
                            {
                                // Part until "_" is the type
                                string typeName = groupName.Remove(groupName.LastIndexOfAny(seperator));
                                ParseType parseType = (ParseType)Enum.Parse(typeof(ParseType), typeName);

                                // Get the value to convert
                                string valueText = match.Groups[groupName].Value;

                                // Try to parse string based on value
                                object target = null;
                                try
                                {
                                    target = Parse(parseType, valueText);
                                }
                                catch (Exception e)
                                {
                                    Trace("scanf_trace: exception:", e.ToString());
                                }
                                finally
                                {
                                    targets.Add(target);
                                }

                                // Terminate early if we have the number of types 
                                // found in the earlier scan
                                if (targets.Count == typeCount)
                                {
                                    return targets.ToArray();
                                }
                            }
                        }
                    }
                }
            }

            // Return what we've got
            return targets.ToArray();
        }

        /// <summary>
        /// Delegate that replaces the format code with the regexp pattern.
        /// </summary>
        /// <param name="m">The match object.</param>
        /// <returns>The updated string.</returns>
        private string ReplaceFormatCode(Match m)
        {
            string groupName;

            // Update the group index
            typeCount++;

            // Determine the group name
            if (types.ContainsKey(m.Value))
            {
                groupName = types[m.Value].ToString();
            }
            else
            {
                // Skip pattern
                groupName = "Null";
            }

            // Assemble the return string
            StringBuilder sb = new StringBuilder();
            sb.Append("(?<");
            sb.Append(groupName);
            sb.Append(seperator[0]);
            sb.Append(typeCount);
            sb.Append(">");
            sb.Append(patterns[m.Value]);
            sb.Append(")");

            return sb.ToString();
        }

        /// <summary>
        /// Parse text into a value by type.
        /// Does not throw and exception but returns null on parse errors.
        /// </summary>
        /// <param name="parseType">The type of object to parse into.</param>
        /// <param name="text">The text to parse.</param>
        /// <returns>The parsed value or null.</returns>
        private object Parse(ParseType parseType, string text)
        {
            switch (parseType)
            {
                case ParseType.Null:
                    return null;
                case ParseType.String:
                    return text;
                case ParseType.Int16:
                    return Int16.Parse(text, CultureInfo.InvariantCulture);
                case ParseType.UInt16:
                    return UInt16.Parse(text, CultureInfo.InvariantCulture);
                case ParseType.UInt16Hex:
                    return UInt16.Parse(text, NumberStyles.HexNumber, CultureInfo.InvariantCulture);
                case ParseType.UInt16Oct:
                    return Convert.ToUInt16(text, 8);
                case ParseType.Int32:
                    return Int32.Parse(text, CultureInfo.InvariantCulture);
                case ParseType.UInt32:
                    return UInt32.Parse(text, CultureInfo.InvariantCulture);
                case ParseType.UInt32Hex:
                    return UInt32.Parse(text, NumberStyles.HexNumber, CultureInfo.InvariantCulture);
                case ParseType.UInt32Oct:
                    return Convert.ToUInt32(text, 8);
                case ParseType.Int64:
                    return Int64.Parse(text, CultureInfo.InvariantCulture);
                case ParseType.UInt64:
                    return UInt64.Parse(text, CultureInfo.InvariantCulture);
                case ParseType.UInt64Hex:
                    return UInt64.Parse(text, NumberStyles.HexNumber, CultureInfo.InvariantCulture);
                case ParseType.UInt64Oct:
                    return Convert.ToUInt64(text, 8);
                case ParseType.Single:
                    return Single.Parse(text, CultureInfo.InvariantCulture);
                case ParseType.Double:
                    return Double.Parse(text, CultureInfo.InvariantCulture);
                case ParseType.Char:
                    return Char.Parse(text);
            }

            return null;
        }

        /// <summary>
        /// Message tracer for debugging.
        /// </summary>
        /// <param name="message">The message to print to the console.</param>
        private static void Trace(string label, string message)
        {
            if (EnableTrace)
            {
                System.Console.Write(label);
                System.Console.Write(": ");
                System.Console.WriteLine(message);
            }
        }
    }
}
