﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Serialization;

namespace NewGamePhysics.Utilities
{
    /// <summary>
    /// Implements a replacement for System.Uri which can be serialized.
    /// </summary>
    public sealed class SerializableUri : IXmlSerializable
    {
        /// <summary>
        /// The underlying Uri instance.
        /// </summary>
        private Uri _uri = null;

        /// <summary>
        /// Gets or sets the Uri instance.
        /// </summary>
        public Uri Uri
        {
            get
            {
                return this._uri;
            }
            set
            {
                this._uri = value;
            }
        }

        /// <summary>
        /// Default constructor.
        /// </summary>
        public SerializableUri()
        {
        }

        /// <summary>
        /// Constructor with uri parameter.
        /// </summary>
        /// <param name="uri">The uri to represent.</param>
        public SerializableUri(Uri uri)
        {
            if (uri == null)
            {
                throw new ArgumentNullException("uri", "Uri cannot be null.");
            }

            this._uri = uri;
        }

        /// <summary>
        /// Constructor with string parameter.
        /// </summary>
        /// <param name="uri">The uri to create.</param>
        public SerializableUri(string uri)
        {
            if (String.IsNullOrEmpty(uri))
            {
                throw new ArgumentNullException("uri", "Uri cannot be null.");
            }

            this._uri = new Uri(uri);
        }

        /// <summary>
        /// Convert between serializable and system form.
        /// </summary>
        /// <param name="uri">The serializable uri.</param>
        /// <returns>The system Uri</returns>
        public static implicit operator Uri(SerializableUri uri)
        {
            return uri.Uri;
        }

        /// <summary>
        /// Convert between system and serializable form.
        /// </summary>
        /// <param name="uri">The system Uri</param>
        /// <returns>The serializable Uri</returns>
        public static implicit operator SerializableUri(Uri uri)
        {
            return new SerializableUri(uri);
        }

        #region IXmlSerializable_members

        /// <summary>
        /// Schema for SerializableUri.
        /// </summary>
        /// <returns>Always return null.</returns>
        System.Xml.Schema.XmlSchema IXmlSerializable.GetSchema()
        {
            return null;
        }

        /// <summary>
        /// Read a serializable Uri element.
        /// </summary>
        /// <param name="reader">The Xml reader to read from.</param>
        void IXmlSerializable.ReadXml(System.Xml.XmlReader reader)
        {
            try
            {
                this._uri = new Uri(reader.ReadString());
            }
            catch (Exception)
            {
                this._uri = null;
            }
        }

        /// <summary>
        /// Write a serializable uri element.
        /// </summary>
        /// <param name="writer">The Xml writer to write to.</param>
        void IXmlSerializable.WriteXml(System.Xml.XmlWriter writer)
        {
            writer.WriteString((this._uri != null) ? this._uri.AbsoluteUri : "");
        }

        #endregion

        /// <summary>
        /// Convert a serializable Uri instance to a string.
        /// </summary>
        /// <returns>The absolute Uri or null.</returns>
        public override string ToString()
        {
            return (this._uri != null) ? this._uri.AbsoluteUri : null;
        }
    }
}
