﻿#region File Description
//-----------------------------------------------------------------------------
// TextureHelpers.cs
//-----------------------------------------------------------------------------
#endregion

namespace NewGamePhysics.Utilities
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;

    /// <summary>
    /// Creates single color textures for drawing rectangles.
    /// </summary>
    public class TextureHelpers
    {
        /// <summary>
        /// Creates single color textures for drawing rectangles.
        /// Default creares a 1x1 pixel black texture.
        /// </summary>
        /// <param name="graphicsDevice">The graphics device to use.</param>
        /// <returns>The newly created texture.</returns>
        public static Texture2D CreateColorTexture(GraphicsDevice graphicsDevice)
        {
            return Create(graphicsDevice, 1, 1, new Color());
        }

        /// <summary>
        /// Creates single color textures for drawing rectangles.
        /// Creates a 1x1 pixel texture of the specified color.
        /// </summary>
        /// <param name="graphicsDevice">The graphics device to use.</param>
        /// <param name="color">The color to set the texture to.</param>
        /// <returns>The newly created texture.</returns>
        public static Texture2D Create(GraphicsDevice graphicsDevice, Color color)
        {
            return Create(graphicsDevice, 1, 1, color);
        }

        /// <summary>
        /// Creates single color textures for drawing rectangles.
        /// Creates a texture of the specified color.
        /// </summary>
        /// <param name="graphicsDevice">The graphics device to use.</param>
        /// <param name="width">The width of the texture.</param>
        /// <param name="height">The height of the texture.</param>
        /// <param name="color">The color to set the texture to.</param>
        /// <returns>The newly created texture.</returns>
        public static Texture2D Create(GraphicsDevice graphicsDevice, int width, int height, Color color)
        {
            // create the rectangle texture without colors
            Texture2D texture = new Texture2D(
                graphicsDevice, 
                width, 
                height, 
                1, 
                TextureUsage.None,
                SurfaceFormat.Color);

            // Create a color array for the pixels
            Color[] colors = new Color[width * height];
            for (int i = 0; i < colors.Length; i++)
            {
                colors[i] = new Color(color.ToVector3());
            }

            // Set the color data for the texture
            texture.SetData(colors);

            return texture;
        }

        /// <summary>
        /// Convert a texture into a 1-dimensionsional list of colors
        /// </summary>
        /// <param name="texture">The texture to convert.</param>
        /// <returns>The list of colors (1D array).</returns>
        public static Color[] TextureToColors(Texture2D texture)
        {
            Color[] color1D = new Color[texture.Width * texture.Height];
            texture.GetData(color1D);
            return color1D;
        }

        /// <summary>
        /// Convert a texture into a 2-dimensionsional color array.
        /// </summary>
        /// <param name="texture">The texture to convert.</param>
        /// <returns>The 2D color array.</returns>
        public static Color[,] TextureToColorArray(Texture2D texture)
        {
            // Get colors as list
            Color[] color1D = TextureToColors(texture);

            // Create new 2D array for colors
            Color[,] color2D = new Color[texture.Width, texture.Height];

            // Copy color from list to array
            int x=0; 
            int y=0;
            foreach (Color color in color1D)
            {
                color2D[x, y] = color;
                x++;
                if (x > texture.Width)
                {
                    y++;
                    x = 0;
                }
            }

            return color2D;
        }
    }
}
