﻿//-----------------------------------------------------------------------------
// PendulumGame.cs
// (A. Schiffler, 2009-2011)
//-----------------------------------------------------------------------------

namespace PendulumGame
{
    using System;
    using System.Threading;
    using System.Collections.Generic;
    using System.Text;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Graphics;
    using NewGamePhysics.Networking;
    using NewGamePhysics.StateManager;

    /// <summary>
    /// This is the main type for your game
    /// </summary>
    public class PendulumGame : Microsoft.Xna.Framework.Game
    {
        /// <summary>
        /// The agent name for PlayTrulyRandom.
        /// </summary>
        public const string PlayTrulyRandomAgentName = "PendulumGame_v3";

        /// <summary>
        /// The current game state.
        /// </summary>
        private static PendulumGameState state;

        /// <summary>
        /// Transmitter for info links.
        /// </summary>
        private InfoLinkTransmitter infoLinkTransmitter;

        /// <summary>
        /// Webservice to submit in-game generated entropy.
        /// </summary>
        private PlayTrulyRandom playTrulyRandomWebservice;

        /// <summary>
        /// Info link collections on various topics.
        /// </summary>
        private Dictionary<string, InfoLink[]> infoLinkCollections;

        /// <summary>
        /// Global screen manager
        /// </summary>
        private ScreenManager screenManager;

        /// <summary>
        /// Constructor of pendulum game.
        /// </summary>
        public PendulumGame()
        {
            // Parse command-line parameters
            bool fullscreen = false;
            bool oneGamepad = false;
            bool disableShaders = false;
            bool xboxResolution = false;
            bool audioOnDemand = false;
            string[] args = System.Environment.GetCommandLineArgs();
            if (args.Length > 1)
            {
                for (int i = 1; i < args.Length; i++)
                {
                    if (!string.IsNullOrEmpty(args[i]))
                    {
                        if (args[i].Equals("/fullscreen", StringComparison.InvariantCultureIgnoreCase))
                        {
                            fullscreen = true;
                        }
                        else if (args[i].Equals("/onegamepad", StringComparison.InvariantCultureIgnoreCase))
                        {
                            oneGamepad = true;
                        }
                        else if (args[i].Equals("/twogamepads", StringComparison.InvariantCultureIgnoreCase))
                        {
                            oneGamepad = false;
                        }
                        else if (args[i].Equals("/noshaders", StringComparison.InvariantCultureIgnoreCase))
                        {
                            disableShaders = true;
                        }
                        else if (args[i].Equals("/xboxscreen", StringComparison.InvariantCultureIgnoreCase))
                        {
                            xboxResolution = true;
                        }
                        else if (args[i].Equals("/audioondemand", StringComparison.InvariantCultureIgnoreCase))
                        {
                            audioOnDemand = true;
                        }
                        else
                        {
                            System.Console.WriteLine("PendulumGame [/fullscreen] [/xboxscreen] [/noshaders] [/onegamepad|/twogamepads] [/audioondemand]");
                        }
                    }
                }
            }

            // Create game state
            state = new PendulumGameState();

            // Set content root
            Content.RootDirectory = "Content";

            // Setup graphics
            GraphicsDeviceManager graphics = new GraphicsDeviceManager(this);

            // Designed for a resolution suitable for both Windows and XBox 360
            if (xboxResolution)
            {
                // Xbox
                graphics.PreferredBackBufferWidth = 853;
                graphics.PreferredBackBufferHeight = 480;
                PendulumGame.State.Scale = new Vector2(1.0f, 1.0f);
            }
            else
            {
                // XGA
                graphics.PreferredBackBufferWidth = 1024;
                graphics.PreferredBackBufferHeight = 768;
                PendulumGame.State.Scale = new Vector2(
                    (float)1024 / (float)853,
                    (float)768 / (float)480);
            }

            // Maybe enable fullscreen mode
            graphics.IsFullScreen = fullscreen;

            // Enable antialiasing
            graphics.PreferMultiSampling = true;

            // Enable on-demand real-time audio
            PendulumGame.State.RealtimeAudioOnDemand = audioOnDemand;

            // Store gamepad choice
            PendulumGame.State.UseOnlyOneGamepad = oneGamepad;

            // Store shader choice
            PendulumGame.State.DisablePixelShaderEffects = disableShaders;

            // Enable mouse
            this.IsMouseVisible = true;

            // Create the screen manager component.
            screenManager = new ScreenManager(this);
            Components.Add(screenManager);
        }

        /// <summary>
        /// Gets the current game state for PendulumGame.
        /// </summary>
        public static PendulumGameState State
        {
            get { return PendulumGame.state; }
        }

        /// <summary>
        /// Initialize: load base initialize/
        /// </summary>
        protected override void Initialize()
        {
            base.Initialize();
        }

        /// <summary>
        /// LoadContent: load some common content used across all of pendulum game.
        /// </summary>
        protected override void LoadContent()
        {
            // Load textures
            screenManager.Textures.Add("rivet", Content.Load<Texture2D>("Sprites/rivet"));
            screenManager.Textures.Add("metal", Content.Load<Texture2D>("Sprites/metal_texture"));
            //
            screenManager.Textures.Add("apple", Content.Load<Texture2D>("Sprites/red_apple"));
            screenManager.Textures.Add("copperball", Content.Load<Texture2D>("Sprites/copperball"));
            screenManager.Textures.Add("silverball", Content.Load<Texture2D>("Sprites/silverball"));
            screenManager.Textures.Add("goldball", Content.Load<Texture2D>("Sprites/goldball"));
            screenManager.Textures.Add("duck", Content.Load<Texture2D>("Sprites/yellow_duck"));

            // Load fonts
            screenManager.Fonts.Add("retro", Content.Load<SpriteFont>("Fonts/retroMedium"));
            screenManager.Fonts.Add("courier", Content.Load<SpriteFont>("Fonts/courier"));

            // Load info links
            string fileContent;
            this.infoLinkCollections = new Dictionary<string, InfoLink[]>();
            string[] infoLinkTopics = { "Gravity", "DoublePendulum", "Randomness" };
            foreach (string topic in infoLinkTopics)
            {
                fileContent =
                    System.IO.File.ReadAllText(@"InfoLinkData\" + topic + ".xml", Encoding.Unicode);
                if (!string.IsNullOrEmpty(fileContent))
                {
                    InfoLink[] infoLinks =
                        InfoLinkSerializer.DeserializeArray(fileContent);
                    if (infoLinks != null)
                    {
                        this.infoLinkCollections.Add(topic, infoLinks);
                    }
                }
            }

            // Open webservice link
            this.playTrulyRandomWebservice = new PlayTrulyRandom(
                    PlayTrulyRandomAgentName);
            this.playTrulyRandomWebservice.RegisterAgent();

            // Transmitter for info links
            infoLinkTransmitter = new InfoLinkTransmitter();
            infoLinkTransmitter.StartTransmitter();

            // Activate the first screens.
            screenManager.AddScreen(new PendulumBackgroundScreen(), null);
            screenManager.AddScreen(new PendulumMainMenuScreen(), null);
        }

        /// <summary>
        /// Unload all pendulum game content.
        /// </summary>
        protected override void UnloadContent()
        {
            // Shut down transmitter
            infoLinkTransmitter.StopTransmitter();

            base.UnloadContent();
        }

        /// <summary>
        /// Default updater: calls base update.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Update(GameTime gameTime)
        {
            base.Update(gameTime);
        }

        /// <summary>
        /// Default draw: black screen.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Draw(GameTime gameTime)
        {
            // Clear to black
            this.GraphicsDevice.Clear(Color.Black);

            // Redraw
            base.Draw(gameTime);
        }

        /// <summary>
        /// Pick random info link for a topic and send it
        /// </summary>
        /// <param name="topic">The topic to choose.</param>
        internal void SendRandomInfoLink(string topic)
        {
            if (this.infoLinkCollections.ContainsKey(topic))
            {
                InfoLink[] infoLinks = this.infoLinkCollections[topic];
                if ((infoLinks != null) && (infoLinks.Length > 0))
                {
                    int randomPick =
                        PendulumGame.state.PhysicalRandomNumberGenerator.Next(
                            0,
                            infoLinks.Length - 1);
                    this.infoLinkTransmitter.Transmit(infoLinks[randomPick]);
                }
            }
        }

        /// <summary>
        /// Submit a bit string (0 and 1 characters) to the PTR webservice
        /// as background process.
        /// </summary>
        /// <param name="entropy">Bitstring representing new entropy.</param>
        internal void SubmitEntropyBits(string entropy)
        {
            Thread thread;
            thread = new System.Threading.Thread(
                this.playTrulyRandomWebservice.OnlineSubmit);
            thread.Start(entropy.Clone());
        }
    }
}

