﻿//-----------------------------------------------------------------------------
// PendulumGameState.cs
// (A. Schiffler, 2009)
//-----------------------------------------------------------------------------

namespace PendulumGame
{
    using System;
    using Microsoft.Xna.Framework;
    using NewGamePhysics.Physics;
    using NewGamePhysics.PhysicalElements;
    using NewGamePhysics.Networking;

    /// <summary>
    /// The global state of the pendulum game
    /// manipulated by the players.
    /// </summary>
    public class PendulumGameState
    {
        /// <summary>
        /// Number of players in the game.
        /// </summary>
        public const int NumPlayers = 2;

        /// <summary>
        /// The game player state.
        /// </summary>
        private GamePlayer[] players = 
            new GamePlayer[NumPlayers];

        /// <summary>
        ///  Celestial object we are using for gravity.
        /// </summary>
        private CelestialObject currentCelestialObject = 
            CelestialObject.Earth;

        /// <summary>
        /// The earths gravity model we are using.
        /// </summary>
        private EarthGravityModel currentEarthGravityModel = 
            EarthGravityModel.GfcGrid;

        /// <summary>
        /// The mars gravity model we are using.
        /// </summary>
        private MarsGravityModel currentMarsGravityModel =
            MarsGravityModel.Normal;

        /// <summary>
        /// The current texture variation to use for a planet.
        /// </summary>
        private int currentTextureVariation = 0;

        /// <summary>
        /// The currently selected gravity value.
        /// </summary>
        private double currentGravity = 9.81;

        /// <summary>
        /// The rotational friction to use for the hinges.
        /// </summary>
        private RotationalFrictionType currentRotationalFrictionType =
            RotationalFrictionType.Stribeck;

        /// <summary>
        /// The physical random number generator for the game.
        /// </summary>
        private PhysicalRandomNumberGenerator physicalRandomNumberGenerator;

        /// <summary>
        /// Counts the number of entropy bits that were submitted.
        /// </summary>
        private int submittedEntropyBits;

        /// <summary>
        /// Flag indicating that only one gamepad is connected.
        /// </summary>
        private bool useOnlyOneGamepad;

        /// <summary>
        /// Flag indicating that pixel shader effects should be disabled.
        /// </summary>
        private bool disablePixelShaderEffects;

        /// <summary>
        /// Flag indicating that realtime audio is on-demand (button press) only.
        /// </summary>
        private bool realtimeAudioOnDemand;

        /// <summary>
        /// The entropy source type to use for random numbers.
        /// </summary>
        private EntropySourceType currentEntropySourceType;

        /// <summary>
        /// The 2D scaling matrix for design-to-screen resizing.
        /// </summary>
        private Vector2 scale;

        /// <summary>
        /// Creates and initializes a pendulum game state.
        /// </summary>
        public PendulumGameState()
        {
            this.physicalRandomNumberGenerator =
                new PhysicalRandomNumberGenerator(
                    EntropySourceType.PlayTrulyRandom,
                    PendulumGame.PlayTrulyRandomAgentName);
        }

        /// <summary>
        /// Gets the scaling matrix for design-to-screen resizing.
        /// </summary>
        public Vector2 Scale
        {
            get { return this.scale; }
            set { this.scale = value;  }
        }

        /// <summary>
        ///  Gets or sets the celestial object we are modelling.
        /// </summary>
        public CelestialObject CurrentCelestialObject
        {
            get { return this.currentCelestialObject; }
            set { this.currentCelestialObject = value; }
        }

        /// <summary>
        /// Gets or sets the earths gravity model we are using.
        /// </summary>
        public EarthGravityModel CurrentEarthGravityModel
        {
            get { return this.currentEarthGravityModel; }
            set { this.currentEarthGravityModel = value; }
        }

        /// <summary>
        /// Gets or sets the mars gravity model we are using.
        /// </summary>
        public MarsGravityModel CurrentMarsGravityModel
        {
            get { return this.currentMarsGravityModel; }
            set { this.currentMarsGravityModel = value; }
        }

        /// <summary>
        /// Gets or sets the texture variation to use .
        /// </summary>
        public int CurrentTextureVariation
        {
            get { return this.currentTextureVariation; }
            set { this.currentTextureVariation = value; }
        }

        /// <summary>
        /// Gets or sets the current rotational friction type.
        /// </summary>
        public RotationalFrictionType CurrentRotationalFrictionType
        {
            get { return this.currentRotationalFrictionType; }
            set { this.currentRotationalFrictionType = value; }
        }

        /// <summary>
        /// Gets or sets the entropy source type to use for random numbers.
        /// </summary>
        public EntropySourceType CurrentEntropySourceType
        {
            get { return this.currentEntropySourceType; }
            set { this.currentEntropySourceType = value; }
        }

        /// <summary>
        /// Gets or sets the current gravity value.
        /// </summary>
        public double CurrentGravity
        {
            get { return this.currentGravity; }
            set { this.currentGravity = value; }
        }

        /// <summary>
        /// Gets or sets the physical random number generator.
        /// </summary>
        public PhysicalRandomNumberGenerator PhysicalRandomNumberGenerator
        {
            get { return this.physicalRandomNumberGenerator; }
            set { this.physicalRandomNumberGenerator = value; }
        }

        /// <summary>
        /// Gets or Sets the count of entropy bits that were submitted.
        /// </summary>
        public int SubmittedEntropyBits
        {
            get { return this.submittedEntropyBits; }
            set { this.submittedEntropyBits = value; }
        }

        /// <summary>
        /// Gets or sets a flag indicating whether only one gamepad is connected.
        /// </summary>
        public bool UseOnlyOneGamepad
        {
            get { return this.useOnlyOneGamepad; }
            set { this.useOnlyOneGamepad = value; }
        }

        /// <summary>
        /// Gets or sets a flag indicating whether real-time audio should be on deman
        /// (i.e. by button press only).
        /// </summary>
        public bool RealtimeAudioOnDemand
        {
            get { return this.realtimeAudioOnDemand; }
            set { this.realtimeAudioOnDemand = value; }
        }

        /// <summary>
        /// Gets or sets a flag indicating whether the pixel shader effects should be used
        /// on the gameplay screen.
        /// </summary>
        public bool DisablePixelShaderEffects
        {
            get { return this.disablePixelShaderEffects; }
            set { this.disablePixelShaderEffects = value; }
        }

        /// <summary>
        /// Gets the players
        /// </summary>
        public GamePlayer[] Players
        {
            get { return this.players; }
        }
    }
}
