﻿//-----------------------------------------------------------------------------
// PendulumGravityChooserScreen.cs
//-----------------------------------------------------------------------------

namespace PendulumGame
{
    using System;
    using System.Threading;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Audio;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Physics;
    using NewGamePhysics.GraphicalElements;
    using NewGamePhysics.PhysicalElements;

    /// <summary>
    /// This screen implements gravity chooser for the Pendulum Game.
    /// </summary>
    class PendulumGravityChooserScreen : GameScreen
    {
        #region Fields

        /// <summary>
        /// The content manager.
        /// </summary>
        private ContentManager contentManager;

        /// <summary>
        /// The animated gravity selector shown on this screen.
        /// </summary>
        private GravitySelector gravitySelector;

        /// <summary>
        /// Speed change from user input.
        /// </summary>
        private double speedStep = 0.1;

        /// <summary>
        /// Maximum speed.
        /// </summary>
        private double speedMax = 1.5;

        /// <summary>
        /// Drag on speed change.
        /// </summary>
        private double speedDragFactor = 0.95;

        /// <summary>
        /// Residual longitudonal rotation speed for user interface.
        /// </summary>
        private double longitudeSpeed = 0.0;

        /// <summary>
        /// Residual latitudinal rotation speed for user interface.
        /// </summary>
        private double latitudeSpeed = 0.0;

        /// <summary>
        /// Flag that triggers additional zooming during fade out.
        /// </summary>
        private bool zoom;

        /// <summary>
        /// Xact audio cue currently playing.
        /// </summary>
        private Cue cue;

        #endregion

        #region Initialization

        /// <summary>
        /// Constructor of the screen.
        /// </summary>
        public PendulumGravityChooserScreen()
        {
            TransitionOnTime = TimeSpan.FromSeconds(1.5);
            TransitionOffTime = TimeSpan.FromSeconds(1.5);

            this.zoom = false;
        }

        /// <summary>
        /// Load graphics content for the game.
        /// </summary>
        public override void LoadContent()
        {
            if (this.contentManager == null)
            {
                this.contentManager = new ContentManager(ScreenManager.Game.Services, "Content");
            }

            // Create physical element(s)
            GravityCalculator gravityCalculator = 
                new GravityCalculator(PendulumGame.State.CurrentCelestialObject);

            // Per-object settings and sound
            this.cue = null;
            switch (PendulumGame.State.CurrentCelestialObject)
            {
                case CelestialObject.Earth:
                    gravityCalculator.EarthGravityModel =
                        PendulumGame.State.CurrentEarthGravityModel;
                    this.cue = this.ScreenManager.SoundBank.GetCue("earth");
                    break;
                case CelestialObject.Mars:
                    gravityCalculator.MarsGravityModel =
                        PendulumGame.State.CurrentMarsGravityModel;
                    break;
                case CelestialObject.Sun:
                    this.cue = this.ScreenManager.SoundBank.GetCue("sun");
                    break;
                case CelestialObject.Venus:
                    this.cue = this.ScreenManager.SoundBank.GetCue("venus");
                    break;
                case CelestialObject.Jupiter:
                    this.cue = this.ScreenManager.SoundBank.GetCue("jupiter");
                    break;
                case CelestialObject.Callisto:
                    this.cue = this.ScreenManager.SoundBank.GetCue("callisto");
                    break;
                case CelestialObject.Ganymede:
                    this.cue = this.ScreenManager.SoundBank.GetCue("ganymede");
                    break;
                case CelestialObject.Europa:
                    this.cue = this.ScreenManager.SoundBank.GetCue("europa");
                    break;
                case CelestialObject.Saturn:
                    this.cue = this.ScreenManager.SoundBank.GetCue("saturn");
                    break;
            }

            this.gravitySelector = new GravitySelector(
                ScreenManager,
                PendulumGame.State.CurrentCelestialObject,
                gravityCalculator);

            // Lower volume of sound
            this.ScreenManager.SetVolume("Default", 1.0f);

            // Start sound
            if (null != this.cue)
            {
                this.cue.Play();
            }

            // Reset game time
            ScreenManager.Game.ResetElapsedTime();
        }

        /// <summary>
        /// Unload graphics content used by the game.
        /// </summary>
        public override void UnloadContent()
        {
            if (null != this.cue)
            {
                this.cue.Stop(AudioStopOptions.Immediate);
            }

            this.contentManager.Unload();
        }

        #endregion

        #region Update and Draw

        /// <summary>
        /// Updates the state of the game. This method checks the GameScreen.IsActive
        /// property, so the game will stop updating when the pause menu is active,
        /// or if you tab away to a different application.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

            if (IsActive)
            {
                // Maybe start sound
                if ((null != this.cue) && (!this.cue.IsPlaying))
                {
                    this.cue.Resume();
                }

                // Animate gravity selector
                gravitySelector.Update(gameTime);
            }
            else
            {
                // Maybe stop sound
                if ((null != this.cue) && (this.cue.IsPlaying))
                {
                    this.cue.Pause();
                }
            }
        }

        /// <summary>
        /// Lets the game respond to player input. Unlike the Update method,
        /// this will only be called when the gameplay screen is active.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            if (input == null)
            {
                throw new ArgumentNullException("input");
            }

            if (input.IsInputCancel(null) || input.IsDisconnected(null))
            {
                ScreenManager.AddScreen(new PendulumPauseMenuScreen(), null);
            }
            else
            {
                if (input.IsInputSelect(null))
                {
                    // Capture selected value
                    PendulumGame.State.CurrentGravity = gravitySelector.Gravity;
                    this.zoom = true;

                    // Quit to options menu
                    PendulumLoadingScreen.Load(
                        ScreenManager,
                        true,
                        null,
                        new PendulumGameplayScreen());
                }

                // Keyboard speed adjustment
                if (input.IsInputLeft(null))
                {
                    longitudeSpeed += speedStep;
                }
                else if (input.IsInputRight(null))
                {
                    longitudeSpeed -= speedStep;
                }

                if (input.IsInputUp(null))
                {
                    latitudeSpeed += speedStep;
                }
                else if (input.IsInputDown(null))
                {
                    latitudeSpeed -= speedStep;
                }

                // Speed limiter
                if (longitudeSpeed < -speedMax)
                {
                    longitudeSpeed = -speedMax;
                }
                else if (longitudeSpeed > speedMax)
                {
                    longitudeSpeed = speedMax;
                }

                if (latitudeSpeed < -speedMax)
                {
                    latitudeSpeed = -speedMax;
                }
                else if (latitudeSpeed > speedMax)
                {
                    latitudeSpeed = speedMax;
                }

                // Update rotation
                gravitySelector.Longitude += longitudeSpeed;
                gravitySelector.Latitude += latitudeSpeed;

                // Drag speed
                longitudeSpeed *= speedDragFactor;
                latitudeSpeed *= speedDragFactor;
            }
        }

        /// <summary>
        /// Draws the gameplay screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            // Match transition to selector size
            if (this.zoom)
            {
                gravitySelector.DisplaySize = 1.0f + TransitionPosition;
            } 
            else 
            {
                gravitySelector.DisplaySize = 1.0f - TransitionPosition;
            }

            // Draw gravity selector
            gravitySelector.Draw(gameTime);

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }
        }

        #endregion
    }
}

