﻿#region File Description
//-----------------------------------------------------------------------------
// PendulumVideoScreen.cs
//-----------------------------------------------------------------------------
#endregion

namespace PendulumGame
{
    using System;
    using Microsoft.Xna.Framework;
    using Microsoft.Xna.Framework.Content;
    using Microsoft.Xna.Framework.Graphics;
    using Microsoft.Xna.Framework.Media;
    using Microsoft.Xna.Framework.Input;

    using NewGamePhysics.StateManager;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Plays a video file
    /// </summary>
    class PendulumVideoScreen : GameScreen
    {
        #region Fields

        /// <summary>
        /// The content manager associated with this screen.
        /// </summary>
        ContentManager content;

        /// <summary>
        /// The video file.
        /// </summary>
        Video videoFile;

        /// <summary>
        /// Flag indicating that the video playback was started.
        /// </summary>
        bool videoStarted = false;

        /// <summary>
        /// The chromakey effect pixel shader.
        /// </summary>
        Effect effect;

        /// <summary>
        /// The reference color for chroma keying.
        /// </summary>
        Color referenceColor;

        #endregion

        #region Initialization

        /// <summary>
        /// Constructor.
        /// </summary>
        public PendulumVideoScreen()
        {
            TransitionOnTime = TimeSpan.FromSeconds(0.5);
            TransitionOffTime = TimeSpan.FromSeconds(0.5);
        }

        /// <summary>
        /// Loads graphics content for this screen. The background texture is quite
        /// big, so we use our own local ContentManager to load it. This allows us
        /// to unload before going from the menus into the game itself, wheras if we
        /// used the shared ContentManager provided by the Game class, the content
        /// would remain loaded forever.
        /// </summary>
        public override void LoadContent()
        {
            if (content == null)
            {
                content = new ContentManager(ScreenManager.Game.Services, "Content");
            }

            // Pick random video
            int videoNumber = PendulumGame.State.PhysicalRandomNumberGenerator.Next(0, 12);

            videoFile = content.Load<Video>(@"Videos\Pendulum" + videoNumber);
            effect = content.Load<Effect>(@"Effects\ColorChromaKey");

            // Determine chromakey reference color by averaging
            // the color over the reference image
            Texture2D backgroundReference = 
                content.Load<Texture2D>(@"Textures\PendulumBackgroundReference");
            Color[] colors = TextureHelpers.TextureToColors(backgroundReference);
            ulong rSum = 0;
            ulong gSum = 0;
            ulong bSum = 0;
            ulong num = (ulong)colors.Length;
            foreach (Color color in colors)
            {
                rSum += (ulong)color.R;
                gSum += (ulong)color.G;
                bSum += (ulong)color.B;
            }

            this.referenceColor = new Color(
                (byte)(rSum / num), 
                (byte)(gSum / num), 
                (byte)(bSum / num));
        }

        /// <summary>
        /// Unloads graphics content for this screen.
        /// </summary>
        public override void UnloadContent()
        {
            content.Unload();
        }


        #endregion

        /// <summary>
        /// Lets the game respond to player input. Unlike the Update method,
        /// this will only be called when the gameplay screen is active.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            if (input == null)
            {
                throw new ArgumentNullException("input");
            }

            if (input.IsInputCancel(null) || input.IsDisconnected(null))
            {
                ScreenManager.AddScreen(new PendulumPauseMenuScreen(), null);
            }
            else
            {
                PlayerIndex playerIndex;

                // Back to main menu
                if (input.IsMenuCancel(null, out playerIndex) ||
                    input.IsMenuSelect(null, out playerIndex))
                {
                    PendulumLoadingScreen.Load(
                        ScreenManager, false, null,
                        new PendulumBackgroundScreen(),
                        new PendulumMainMenuScreen());
                }
            }
        }

        #region Update and Draw

        /// <summary>
        /// Updates the background screen. Unlike most screens, this should not
        /// transition off even if it has been covered by another screen: it is
        /// supposed to be covered, after all! This overload forces the
        /// coveredByOtherScreen parameter to false in order to stop the base
        /// Update method wanting to transition off.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)        
        {
            if (!videoStarted)
            {
                ScreenManager.VideoPlayer.Play(videoFile);
                videoStarted = true;
            }
            else
            {
                if (ScreenManager.VideoPlayer.State == MediaState.Stopped)
                {
                    // Return to main menu
                    PendulumLoadingScreen.Load(ScreenManager, false, null, 
                        new PendulumBackgroundScreen(),
                        new PendulumMainMenuScreen());
                }
            }

            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
            {
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
            }

            base.Update(gameTime, otherScreenHasFocus, false);
        }


        /// <summary>
        /// Draws the background screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            Viewport viewport = ScreenManager.GraphicsDevice.Viewport;
            Rectangle fullscreen = new Rectangle(0, 0, viewport.Width, viewport.Height);
            byte fade = TransitionAlpha;

            // Render the video in it's orginal resolution 
            // to the screen using SpriteBatch
            if (ScreenManager.VideoPlayer.State == MediaState.Playing)
            {
                spriteBatch.Begin(SpriteBlendMode.AlphaBlend, SpriteSortMode.Immediate, SaveStateMode.None);
                effect.Begin();
                float threshold = 0.3f;
                effect.Parameters["referenceColor"].SetValue(this.referenceColor.ToVector3());
                effect.Parameters["threshold"].SetValue(threshold);
                foreach (EffectPass pass in effect.CurrentTechnique.Passes)
                {
                    pass.Begin();
                    spriteBatch.Draw(
                        ScreenManager.VideoPlayer.GetTexture(),
                        new Rectangle(
                            (viewport.Width - videoFile.Width) / 2,
                            (viewport.Height - videoFile.Height) / 2, 
                            videoFile.Width, 
                            videoFile.Height),
                        new Color(fade, fade, fade));
                    pass.End();
                }
                effect.End();
                spriteBatch.End();
            }
        }

        #endregion
    }
}

