﻿namespace Tests
{
    using System;
    using System.Collections.Generic;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using NewGamePhysics.Physics;
    using NewGamePhysics.Utilities;

    /// <summary>
    /// Summary description for UnitTestGamma
    /// </summary>
    [TestClass]
    public class UnitTestGravityEarth
    {
        public UnitTestGravityEarth()
        {
        }

        private TestContext testContextInstance;

        /// <summary>
        ///Gets or sets the test context which provides
        ///information about and functionality for the current test run.
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }

        /// <summary>
        /// Tests the coefficient loader for the earths gravity model.
        /// </summary>
        [TestMethod]
        public void GravityEarthLoadModelFile()
        {
            GravityEarthGfcModel gravityEarth = new GravityEarthGfcModel();
            gravityEarth.LoadGfcModel(@"ggm03s.gfc");
            Assert.AreEqual(gravityEarth.ModelOrder, 180);
            Assert.AreEqual(gravityEarth.CalcOrder, 180);

            gravityEarth.LoadGfcModel(@"eigen5c.gfc");
            Assert.AreEqual(gravityEarth.ModelOrder, 360);
            Assert.AreEqual(gravityEarth.CalcOrder, 360);
        }

        /// <summary>
        /// Tests the grid loader for the earths gravity model.
        /// </summary>
        [TestMethod]
        public void GravityEarthLoadGridFile()
        {
            GravityEarthGfcGrid gravityEarth = new GravityEarthGfcGrid();
            gravityEarth.LoadGravityMap(@"eigen5c.gdf");
            Assert.AreEqual(gravityEarth.Size, 65341);
        }

        /// <summary>
        /// Tests the grid-based interpolation estimator for the earths gravity.
        /// </summary>
        [TestMethod]
        public void GravityEarthCalculateGrid()
        {
            GravityEarthGfcGrid gravityEarth = new GravityEarthGfcGrid();
            gravityEarth.LoadGravityMap(@"eigen5c.gdf");

            // Reference: 
            // ICGEM calculation service
            // model: functional: gravity
            string[] testdata = {

            // model: eigen5c (expect minimal difference)
            "126.0000     37.0000   979914.448336",
            "126.2500     37.0000   979922.352799",
            "126.5000     37.0000   979924.633472",
            "126.7500     37.0000   979922.081383",
            "127.0000     37.0000   979916.785445",
            "126.0000     36.7500   979882.959664",
            "126.2500     36.7500   979888.765359",
            "126.5000     36.7500   979897.746912",
            "126.7500     36.7500   979895.879925",
            "127.0000     36.7500   979883.720262",
            "126.0000     36.5000   979863.306337",
            "126.2500     36.5000   979876.866452",
            "126.5000     36.5000   979877.498516",
            "126.7500     36.5000   979853.527464",
            "127.0000     36.5000   979841.388265",
            "126.0000     36.2500   979849.782964",
            "126.2500     36.2500   979862.262440",
            "126.5000     36.2500   979867.176353",
            "126.7500     36.2500   979823.843720",
            "127.0000     36.2500   979841.700404",
            "126.0000     36.0000   979834.101367",
            "126.2500     36.0000   979840.798833",
            "126.5000     36.0000   979842.036805",
            "126.7500     36.0000   979833.011654",
            "127.0000     36.0000   979829.875387",

            // model: egm2008 (expect some difference) 
            "46.0000     47.0000   980784.237968",
            "46.2500     47.0000   980793.748374",
            "46.5000     47.0000   980777.584929",
            "46.7500     47.0000   980780.749349",
            "47.0000     47.0000   980790.402344",
            "46.0000     46.7500   980763.400078",
            "46.2500     46.7500   980774.219026",
            "46.5000     46.7500   980777.852799",
            "46.7500     46.7500   980776.626910",
            "47.0000     46.7500   980785.249340",
            "46.0000     46.5000   980748.768039",
            "46.2500     46.5000   980750.621451",
            "46.5000     46.5000   980755.220104",
            "46.7500     46.5000   980760.313200",
            "47.0000     46.5000   980772.486144",
            "46.0000     46.2500   980733.156652",
            "46.2500     46.2500   980741.272033",
            "46.5000     46.2500   980737.550959",
            "46.7500     46.2500   980733.202427",
            "47.0000     46.2500   980739.222344",
            "46.0000     46.0000   980708.255067",
            "46.2500     46.0000   980709.374456",
            "46.5000     46.0000   980716.814933",
            "46.7500     46.0000   980718.081268",
            "47.0000     46.0000   980720.099740"
            };

            Scanf scanf = new Scanf();
            object[] t;
            for (int i = 0; i < testdata.Length; i++)
            {
                t = scanf.Scan(testdata[i], "%lf %lf %lf");
                double lon = (double)t[0];
                double lat = (double)t[1];
                double expected = (double)t[2]*1e-5;
                double value = gravityEarth.Calculate(lat, lon, 0.0);
                double error = Math.Abs(value - expected);
                Console.WriteLine(
                    "gravityEarth.Calculate({0},{1},0.0): expected {2} got {3} error {4}",
                    lat,
                    lon,
                    expected,
                    value,
                    error);
                Assert.IsTrue(error <= 5e-3);
            }
        }

        /// <summary>
        /// Tests the model calculator for the earths gravity.
        /// (uses eigen5c.gcf)
        /// </summary>
        [TestMethod]
        public void GravityEarthCalculateModelA()
        {
            Console.WriteLine("Model: eigen5c.gfc");

            GravityEarthGfcModel gravityEarth = new GravityEarthGfcModel();
            gravityEarth.LoadGfcModel(@"eigen5c.gfc");

            // Use full order
            gravityEarth.CalcOrder = gravityEarth.ModelOrder;

            // Reference: 
            // ICGEM calculation service
            // model: functional: gravity
            string[] testdata = {

            // model: eigen5c
            "126.0000     37.0000   979914.448336",
            "126.2500     37.0000   979922.352799",
            "126.5000     37.0000   979924.633472",
            "126.7500     37.0000   979922.081383",
            "127.0000     37.0000   979916.785445",
            "126.0000     36.7500   979882.959664",
            "126.2500     36.7500   979888.765359",
            "126.5000     36.7500   979897.746912",
            "126.7500     36.7500   979895.879925",
            "127.0000     36.7500   979883.720262",
            "126.0000     36.5000   979863.306337",
            "126.2500     36.5000   979876.866452",
            "126.5000     36.5000   979877.498516",
            "126.7500     36.5000   979853.527464",
            "127.0000     36.5000   979841.388265",
            "126.0000     36.2500   979849.782964",
            "126.2500     36.2500   979862.262440",
            "126.5000     36.2500   979867.176353",
            "126.7500     36.2500   979823.843720",
            "127.0000     36.2500   979841.700404",
            "126.0000     36.0000   979834.101367",
            "126.2500     36.0000   979840.798833",
            "126.5000     36.0000   979842.036805",
            "126.7500     36.0000   979833.011654",
            "127.0000     36.0000   979829.875387"
            };

            Scanf scanf = new Scanf();
            object[] t;
            for (int i = 0; i < testdata.Length; i++)
            {
                t = scanf.Scan(testdata[i], "%lf %lf %lf");
                double lon = (double)t[0];
                double lat = (double)t[1];
                double expected = (double)t[2] * 1e-5;
                double value = gravityEarth.Calculate(lat, lon, 0.0);
                double error = Math.Abs(value - expected);
                Console.WriteLine(
                    "gravityEarth.Calculate({0},{1},0.0): expected {2} got {3} error {4}",
                    lat,
                    lon,
                    expected,
                    value,
                    error);
                // Model calc does not work yet
                // Assert.IsTrue(error <= 5e-3);
            }
        }

        /// <summary>
        /// Tests the model calculator for the earths gravity.
        /// (uses ggm03s.gcf order 120, compares to egm2008)
        /// </summary>
        [TestMethod]
        public void GravityEarthCalculateModelB()
        {
            Console.WriteLine("Model: ggm03s.gfc");

            GravityEarthGfcModel gravityEarth = new GravityEarthGfcModel();
            gravityEarth.LoadGfcModel(@"ggm03s.gfc");

            // Need to limit order for stability
            // Reference: model text, F. Barthelmes private comm.
            gravityEarth.CalcOrder = 120;

            // Reference: 
            // ICGEM calculation service
            // model: functional: gravity
            string[] testdata = {

            // model: egm2008
            "46.0000     47.0000   980784.237968",
            "46.2500     47.0000   980793.748374",
            "46.5000     47.0000   980777.584929",
            "46.7500     47.0000   980780.749349",
            "47.0000     47.0000   980790.402344",
            "46.0000     46.7500   980763.400078",
            "46.2500     46.7500   980774.219026",
            "46.5000     46.7500   980777.852799",
            "46.7500     46.7500   980776.626910",
            "47.0000     46.7500   980785.249340",
            "46.0000     46.5000   980748.768039",
            "46.2500     46.5000   980750.621451",
            "46.5000     46.5000   980755.220104",
            "46.7500     46.5000   980760.313200",
            "47.0000     46.5000   980772.486144",
            "46.0000     46.2500   980733.156652",
            "46.2500     46.2500   980741.272033",
            "46.5000     46.2500   980737.550959",
            "46.7500     46.2500   980733.202427",
            "47.0000     46.2500   980739.222344",
            "46.0000     46.0000   980708.255067",
            "46.2500     46.0000   980709.374456",
            "46.5000     46.0000   980716.814933",
            "46.7500     46.0000   980718.081268",
            "47.0000     46.0000   980720.099740"
            };

            Scanf scanf = new Scanf();
            object[] t;
            for (int i = 0; i < testdata.Length; i++)
            {
                t = scanf.Scan(testdata[i], "%lf %lf %lf");
                double lon = (double)t[0];
                double lat = (double)t[1];
                double expected = (double)t[2] * 1e-5;
                double value = gravityEarth.Calculate(lat, lon, 0.0);
                double error = Math.Abs(value - expected);
                Console.WriteLine(
                    "gravityEarth.Calculate({0},{1},0.0): expected {2} got {3} error {4}",
                    lat,
                    lon,
                    expected,
                    value,
                    error);
                // Model calc does not work yet.
                // Assert.IsTrue(error <= 5e-3);
            }
        }
    }
}
